/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.report.html;

import de.kobv.mable.ErrorCode;
import de.kobv.mable.common.MableProperties;
import de.kobv.mable.mab.extension.ErrorStatistics;
import de.kobv.mable.mab.extension.ErrorType;
import de.kobv.mable.mab.extension.FieldErrorType;
import de.kobv.mable.mab2.modules.DatasetStatistics;
import de.kobv.mable.mab2.modules.FieldStatistics;
import de.kobv.mable.report.chart.PieChart;
import de.kobv.mable.report.chart.PieSlice;
import de.kobv.mable.report.util.FieldErrorComparator;
import de.kobv.mable.report.util.ValueComparator;
import de.kobv.mable.reports.AbstractReport;
import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringEscapeUtils;

/**
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO need output path that contains resources for report
 * TODO generate PDF later
 */
public class ReportSummary extends AbstractReport {

    private FieldStatistics fieldStatistics;

    private DatasetStatistics datasetStatistics;

    private ErrorStatistics errorStatistics;

    private static final int INDEX_FIELD_ID = 0;

    private static final int INDEX_ERROR_COUNT = 1;

    private static final int INDEX_ERROR_CODE = 2;

    private String basePath = "de/kobv/mable/report/html/";

    /**
     *
     * TODO filename and date
     */
    @Override
    public void generate() {
        try {
            copyFile(basePath + "bvblogo.png", "bvblogo.png");
            copyFile(basePath + "kobvlogo.gif", "kobvlogo.gif");
            copyFile(basePath + "kobvlogo2.jpg", "kobvlogo2.jpg");
            copyFile(basePath + "report.css", "report.css");

            Map<String, String> properties = new HashMap<String, String>();

            int datasetsWithErrors = getDatasetsWithErrors();

            String timestampString = MableProperties.getProperty("timestampString").toString();
            String filename = MableProperties.getProperty("filename").toString();

            String svgFilename = filename + "_" + timestampString + "_report.svg";

            OutputStream out = new FileOutputStream(svgFilename);

            PieChart chart = new PieChart(220, 220);

            int datasetsTotal = ( Integer )MableProperties.getProperty("dataset.count");

            PieSlice slice = new PieSlice(datasetsWithErrors);
            slice.setColor("black");
            slice.setLabel(Integer.toString(datasetsWithErrors));
            chart.addSlice(slice);

            slice = new PieSlice(datasetsTotal - datasetsWithErrors);
            slice.setColor("lightgray");
            slice.setLabel(Integer.toString(datasetsTotal - datasetsWithErrors));
            chart.addSlice(slice);

            chart.renderAsSvg(out);

            out.close();

            properties.put("piefile", svgFilename);
            properties.put("timestamp", MableProperties.getProperty("timestamp").toString());
            properties.put("filename.report", MableProperties.getProperty("report.report").toString());
            properties.put("filename.reject", MableProperties.getProperty("report.reject").toString());
            properties.put("filename.errors", MableProperties.getProperty("report.errors").toString());
            properties.put("filename.nomab001", MableProperties.getProperty("report.nomab001").toString());
            properties.put("filename", MableProperties.getProperty("filename").toString());
            properties.put("dataset.count", MableProperties.getProperty("dataset.count").toString());
            properties.put("datasetsWithErrors", Integer.toString(datasetsWithErrors));
            properties.put("datasetstatus", getDatasetStatusHtml());
            properties.put("datasettypes", getDatasetTypesHtml());
            properties.put("dataseterrors", getDatasetErrorsHtml());
            properties.put("fielderrors", getFieldErrorsHtml());
            properties.put("fieldstats", getFieldStatisticsHtml());

            copyAndFilterFile(basePath + "report.html",
                filename + "_" + timestampString + "_summary.html", properties);
        }
        catch (IOException ioe) {
            // TODO do something
        }
    }

    public FieldStatistics getFieldStatistics() {
        return fieldStatistics;
    }

    public void setFieldStatistics(FieldStatistics fieldStatistics) {
        this.fieldStatistics = fieldStatistics;
    }

    protected String getDatasetErrorsHtml() throws IOException {
        StringWriter writer = new StringWriter();

        PrintWriter pout = new PrintWriter(writer);

        Map<ErrorType, Integer> result = errorStatistics.getFormatErrors();

        pout.println("<table id=\"datasetErrors\">");

        int index = 1;

        for (Map.Entry<ErrorType, Integer> entry : result.entrySet()) {
            pout.println("  <tr>");
            pout.println("    <td class=\"count\">" + entry.getValue() + "</td>");
            pout.println("    <td class=\"message\">(#" + entry.getKey() + ") " +
                StringEscapeUtils.escapeHtml(entry.getKey().getMessage()) + "</td>");
            pout.println("  </tr>");
            if (index++ == 10) break;
        }

        pout.println("</table>");

        pout.close();

        return writer.toString();
    }

    protected String getFieldErrorsHtml() throws IOException {
        StringWriter writer = new StringWriter();

        PrintWriter pout = new PrintWriter(writer);

        Map<FieldErrorType, Integer> result = errorStatistics.getFieldErrors();

//        FieldErrorComparator comparator = new FieldErrorComparator();

//        SortedSet<Object[]> sortedFieldErrors = new TreeSet<Object[]>(comparator);

//        sortedFieldErrors.addAll(result);

        pout.println("<table id=\"fieldErrors\">");

        int index = 1;

        for (Map.Entry<FieldErrorType, Integer> fehler : result.entrySet()) {
            FieldErrorType errorType = fehler.getKey();
            pout.println("  <tr>");
            String field = errorType.getBezeichner();
            field = field.replace("' '", "_");
            pout.println("    <td class=\"field\">" + field + "</td>");
            pout.println("    <td class=\"count\">" + fehler.getValue() + "</td>");
            pout.println("    <td class=\"message\">(#" + errorType.getCode() + ") " +
                StringEscapeUtils.escapeHtml(errorType.getMessage()) + "</td>");
            pout.println("  </tr>");
            if (index++ == 10) break;
        }

        pout.println("</table>");

        pout.close();

        return writer.toString();
    }

    protected String getDatasetStatusHtml() throws IOException {
        StringWriter writer = new StringWriter();

        PrintWriter pout = new PrintWriter(writer);

        Map<Character, Integer> status = datasetStatistics.getStatusCount();

        pout.println("<table id=\"status\">");

        for (Map.Entry<Character, Integer> entry : status.entrySet()) {
            pout.println("  <tr>");
            pout.println("    <td>" + entry.getKey() + "</td>");
            pout.println("    <td>" + entry.getValue() + "</td");
            pout.println("  </tr>");
        }

        pout.println("</table>");

        pout.close();

        return writer.toString();
    }

    protected String getDatasetTypesHtml() throws IOException {
        StringWriter writer = new StringWriter();

        PrintWriter pout = new PrintWriter(writer);

        Map<Character, Integer> types = datasetStatistics.getTypeCount();

        pout.println("<table id=\"status\">");

        for (Map.Entry<Character, Integer> entry : types.entrySet()) {
            pout.println("  <tr>");
            pout.println("    <td>" + entry.getKey() + "</td>");
            pout.println("    <td>" + entry.getValue() + "</td");
            pout.println("  </tr>");
        }

        pout.println("</table>");

        pout.close();

        return writer.toString();

    }

    protected String getFieldStatisticsHtml() throws IOException {
        StringWriter writer = new StringWriter();

        PrintWriter pout = new PrintWriter(writer);

        Map<String, Integer> feldStatistik =
            (Map<String, Integer>) fieldStatistics.getFieldCount();

        SortedSet<String> sortedFields = summarizeFieldStatistics();

        pout.println("<table id=\"fields\">");

        int index = 0;

        String[] fields = sortedFields.toArray(new String[0]);

        for (int i = 0; i < 10; i++) {
            pout.println("  <tr>");
            pout.println("    <td class=\"label\">" + fields[i] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 10] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 10]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 20] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 20]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 30] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 30]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 40] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 40]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 50] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 50]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 60] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 60]) + "</td>");
            pout.println("    <td class=\"label\">" + fields[i + 70] + "</td>");
            pout.println("    <td class=\"value\">" + feldStatistik.get(fields[i + 70]) + "</td>");
            pout.println("  </tr>");
        }

        pout.println("</table>");

        writer.close();

        return writer.toString();
    }

    protected SortedSet<String> summarizeFieldStatistics() {
        Map<String, Integer> feldStatistik =
            (Map<String, Integer>) fieldStatistics.getFieldCount();

        ValueComparator comparator = new ValueComparator(feldStatistik);

        SortedSet<String> sortedIndex = new TreeSet<String>(comparator);

        sortedIndex.addAll(feldStatistik.keySet());

        return sortedIndex;
    }

    protected void copyFile(String source, String dest) throws IOException {
        InputStream in = getClass().getClassLoader().getResourceAsStream(source);
        OutputStream out = new FileOutputStream(dest);

        byte[] buf = new byte[4096];
        int bytesRead;

        while ((bytesRead = in.read(buf, 0, buf.length)) != -1) {
            out.write(buf, 0, bytesRead);
        }

        out.close();
        in.close();
    }

    protected void copyAndFilterFile(String source, String dest,
            Map<String, String> properties) throws IOException {

        InputStream in = getClass().getClassLoader().getResourceAsStream(source);

        Reader rin = new InputStreamReader(in);

        StringWriter wpage = new StringWriter();

        char[] cbuf = new char[4096];
        int charsRead;

        while ((charsRead = rin.read(cbuf, 0, cbuf.length)) != -1) {
            wpage.write(cbuf, 0, charsRead);
        }

        wpage.close();

        rin.close();

        String page = wpage.toString();

        // Replace placeholders in page

        for (Map.Entry<String,String> entry : properties.entrySet()) {
            String placeholder = entry.getKey();
            String value = entry.getValue();

            Pattern pattern = Pattern.compile("\\$\\{" + placeholder + "\\}");
            Matcher matcher = pattern.matcher(page);

            page = matcher.replaceAll(value);
        }

        // Write page to file

        Writer out = new FileWriter(dest);

        out.write(page);

        out.close();

    }

    protected int getDatasetsWithErrors() throws IOException {
        String filename = MableProperties.getProperty("report.errors").toString();

        BufferedReader reader = new BufferedReader(new FileReader(filename));

        String line;

        Set<String> datasetIds = new HashSet<String>();

        Pattern pattern = Pattern.compile("(\\w+), #\\d+");

        while ((line = reader.readLine()) != null) {
            Matcher matcher = pattern.matcher(line);

            if (matcher.find()) {
                String datasetId = matcher.group(1);
                datasetIds.add(datasetId);
            }
        }

        return datasetIds.size();
    }

    public DatasetStatistics getDatasetStatistics() {
        return datasetStatistics;
    }

    public void setDatasetStatistics(DatasetStatistics datasetStatistics) {
        this.datasetStatistics = datasetStatistics;
    }

    public ErrorStatistics getErrorStatistics() {
        return errorStatistics;
    }

    public void setErrorStatistics(ErrorStatistics errorStatistics) {
        this.errorStatistics = errorStatistics;
    }

}
