/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.extension.DatasetError;
import de.kobv.mable.mab.extension.ErrorType;
import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;

import java.util.List;

/**
 * Unit Tests for RequiredCategory class.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class RequiredCategoryTest {

    /**
     * Valid MAB dataset header for testing.
     */
    public static final String VALID_KENNUNG = "01024nM2.01200024      h";

    private MabSatz satz;

    private RequiredCategory module;

    private MockErrorStatistics errorStatistics;

    @Before
    public void before() {
        satz = new MabSatz(VALID_KENNUNG);
        module = new RequiredCategory();
        errorStatistics = new MockErrorStatistics();
        module.setErrorStatistics(errorStatistics);
    }

    @Test
    public void testCheckForKategorieFoundTrue() {
        satz.addFeld(new MabFeld(1, ' '));
        module.setFeldnummer(1);

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(0, errors.size());
    }

    @Test
    public void testCheckForKategorieFoundFalse() {
        module.setFeldnummer(1);

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(1, errors.size());
        assertTrue(errors.get(0) != null);
        assertTrue(errors.get(0).getErrorType() instanceof RequiredCategoryErrorType);
    }

    @Test
    public void testCheckForKategoryAndIndikatorFoundTrue() {
        satz.addFeld(new MabFeld(26, 'a'));
        module.setFeldnummer(26);
        module.setIndikator('a');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(0, errors.size());
    }

    @Test
    public void testCheckForKategoryAndIndikatorFoundFalse() {
        satz.addFeld(new MabFeld(26, 'a'));
        module.setFeldnummer(26);
        module.setIndikator('b');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(1, errors.size());
        assertTrue(errors.get(0) != null);
        assertTrue(errors.get(0).getErrorType() instanceof RequiredCategoryErrorType);
    }

    @Test
    public void testCheckForKategoryAndSatztypFoundTrue() {
        satz.addFeld(new MabFeld(89, ' '));

        module.setFeldnummer(89);
        module.setSatzTyp('h');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(0, errors.size());
    }

    @Test
    public void testCheckForKategoryAndSatztypFoundFalse() {
        module.setFeldnummer(89);
        module.setSatzTyp('h');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(1, errors.size());
        assertTrue(errors.get(0) != null);
        assertTrue(errors.get(0).getErrorType() instanceof RequiredCategoryErrorType);
    }

    @Test
    public void testCheckForKategoryAndWrongSatztypFoundFalse() {
        module.setFeldnummer(89);
        module.setSatzTyp('u');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(0, errors.size());
    }

    @Test
    public void testCheckForKategoryAndIndikatorAndSatztypFoundTrue() {
        satz.addFeld(new MabFeld(89, 'a'));
        module.setFeldnummer(89);
        module.setIndikator('a');
        module.setSatzTyp('h');

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(0, errors.size());
    }

    @Test
    public void testCheckErrorPropertiesSet() {
        satz.addFeld(new MabFeld(1, ' ', "ID0001"));

        module.setFeldnummer(89);

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(1, errors.size());
        assertTrue(errors.get(0) != null);
        assertTrue(errors.get(0).getErrorType() instanceof RequiredCategoryErrorType);

        DatasetError error = errors.get(0);

        assertEquals("ID0001", error.getSatzId());
        assertEquals("089", (( RequiredCategoryErrorType )error.getErrorType() ).getCategory());
    }

    @Test
    public void testReportErrorWithMissingId() {
        module.setFeldnummer(89);

        module.process(satz);

        List<DatasetError> errors = errorStatistics.getErrors();

        assertEquals(1, errors.size());
        assertTrue(errors.get(0) != null);
        assertTrue(errors.get(0).getErrorType() instanceof RequiredCategoryErrorType);

        DatasetError error = errors.get(0);

        assertEquals(null, error.getSatzId());
        assertEquals("089", (( RequiredCategoryErrorType )error.getErrorType() ).getCategory());
    }

    /**
     * TODO improve
     */
    @Test
    public void testToString() {
        assertNotNull(module.toString());
    }

    @Test
    public void testGetMessageDefault() {
        module.setFeldnummer(1);
        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals("Satz ohne MAB-Feld 001 (Identifikationsnummer des Datensatzes).", errorType.getMessage());
    }

    @Test
    public void testGetErrorCodeGenerated() {
        module.setFeldnummer(1);
        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertTrue(errorType.getCode() >= 5000);
    }

    @Test
    public void testConfigureErrorCode() {
        module.setFeldnummer(1);
        module.setErrorCode(2003);

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals(2003, errorType.getCode());
    }

    @Test
    public void testConfigureErrorCodeAndMessageKey() {
        module.setFeldnummer(1);
        module.setErrorCode(2003);
        module.setErrorMessageKey("SATZ_OHNE_MAB001");

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals(2003, errorType.getCode());
        assertEquals("Der Satz hat keine ID, es fehlt ein MAB-Feld 001.", errorType.getMessage());
    }

    @Test
    public void testConfigureErrorMessageKey() {
        module.setFeldnummer(1);
        module.setErrorMessageKey("SATZ_OHNE_MAB001");
        module.setErrorCode(2003);

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals("Der Satz hat keine ID, es fehlt ein MAB-Feld 001.", errorType.getMessage());
    }

}
