/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.IMabFeld;
import de.kobv.mable.mab.MabFeldDefinition;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.MabSatzkennung;
import de.kobv.mable.mab.extension.DatasetError;
import de.kobv.mable.mab.extension.ErrorType;
import de.kobv.mable.mab.extension.IErrorStatistics;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

/**
 * Created with IntelliJ IDEA.
 * User: jens
 * Date: 9/7/12
 * Time: 10:08 AM
 * To change this template use File | Settings | File Templates.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO Felder, die mit einem bestimmten Indikatorpflicht sind (70a)
 */
public class Wiederholfaktoren extends AbstractMableModule2 {

    private IErrorStatistics errorStatistics;

    // TODO Vordefinierte Fehlertypen mit Code und Message
    private Set<ErrorType> errorTypes;

    @Override
    public void process(MabSatz satz) {
        Set<String> kategorien = satz.getKategorien();

        if (kategorien == null) {
            verifyRequiredFields(satz);
        }
        else {
            verifyRequiredFields(satz);
            verifyExistingFields(satz, kategorien);
        }
    }

    public void setErrorStatistics(IErrorStatistics errorStatistics) {
        this.errorStatistics = errorStatistics;
    }

    public void setErrorTypes(final Set<ErrorType> errorTypes) {
        this.errorTypes = errorTypes;
    }

    /**
     * Prüft ob Pflichtfelder vorhanden sind.
     *
     * @param satz IMabSatz
     */
    protected void verifyRequiredFields(MabSatz satz) {
        MabSatzkennung satzkennung = satz.getSatzkennung();

        String satztyp = Character.toString(satzkennung.getTyp());

        for (MabFeldDefinition feldDef : EnumSet.allOf(MabFeldDefinition.class)) {
            if (feldDef.isRequiredForSatztyp(satztyp) && !satz.isFeldPresent(feldDef.getFeldnummerString())) {
                ErrorType errorType = new RequiredCategoryErrorType(feldDef.getFeldnummerString(), satztyp);
                errorStatistics.addError(new DatasetError(errorType, satz.getId()));
            }
        }
    }

    /**
     * Prüft Häufigkeit von gefundenen Feldern.
     *
     * @param satz IMabSatz
     * @param kategorien Gefundene MAB2 Kategorien
     */
    protected void verifyExistingFields(MabSatz satz, Set<String> kategorien) {
        for (String kategorie : kategorien) {
            int feldnummer = 0;

            try {
                feldnummer = Integer.parseInt(kategorie.substring(0, 3));
            }
            catch (NumberFormatException nfe) {
                // Nicht MAB2 Felder werden nicht geprüft
                continue;
            }

            MabFeldDefinition feldDef = MabFeldDefinition.getFeld(feldnummer);

            if (feldDef == null) {
                // Nicht MAB2 Felder werden nicht geprüft
                continue;
            }

            List<IMabFeld> felder = satz.getFelder(feldDef);

            int wiederholfaktor = feldDef.getWiederholfaktor();
            if (felder != null && felder.size() > 0) {
                if (feldDef.isIndikatorWiederholbar()) {
                    if (felder.size() > wiederholfaktor && wiederholfaktor != 0) {
                        // Feld kommt zu häufig vor
                        errorStatistics.addError(new DatasetError(
                                new FieldCountErrorType(feldDef.getFeldnummerString()),
                                satz.getId()
                        ));
                    }
                }
                else {
                    // Jeder Indikator darf nur einmal vorkommen.
                    char[] indikatoren = feldDef.getIndikatoren();

                    for (int i = 0; i < indikatoren.length; i++) {
                        List<IMabFeld> felderMitIndikator = satz.getFelder(feldDef.getFeldnummerString(),
                                indikatoren[i]);
                        if (felderMitIndikator != null && felderMitIndikator.size() > 1) {
                            // Feld mit Indikator kommt mehr als einmal vor
                            errorStatistics.addError(new DatasetError(
                                    new FieldCountErrorType(feldDef.getFeldnummerString(),
                                            Character.toString(indikatoren[i])),
                                    satz.getId()
                            ));
                        }
                    }
                }

                // Darf das Feld überhaupt vorkommen
                Set<String> satztypen = feldDef.getSatztypen();
                String satztyp = Character.toString(satz.getSatzkennung().getTyp());
                if (satztypen != null && !satztypen.contains(satztyp)) {
                    errorStatistics.addError(new DatasetError(
                            new BadExtraFieldErrorType(satztyp, feldDef.getFeldnummerString()),
                            satz.getId()
                    ));

                }
            }
        }
    }

}
