/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.IMabFeld;
import de.kobv.mable.mab.IMabSatz;
import de.kobv.mable.mab.MabFeldDefinition;
import de.kobv.mable.mab.MabSatz;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.validator.routines.ISBNValidator;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Validates ISBN values.
 *
 * This module validates ISBN values in MAB2 fields and detects a number of
 * different errors.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO can this class be reused for ISMN?
 * TODO what about "isbn" anstatt "ISBN"?
 * TODO Fehlertypen definieren
 * TODO Prüfe 540_ und melde in Statistik
 * TODO Mehrere ISBNs in einem Feld? Beispiele?
 *
 * ValidateIsbn produziert IsbnError Instanzen um Fehler zu melden. Dabei kann es sich, um unterschiedliche Fehler
 * handeln.
 *
 */
public class ValidateIsbn extends AbstractMabCheck {

    /**
     * Logger for this class.
     */
    private static final Logger LOG = Logger.getLogger(ValidateIsbn.class);

    /**
     * Validator for ISBN validation.
     */
    private ISBNValidator validator;

    /**
     * Pattern for matching ISBN values.
     */
    private Pattern isbnPattern;

    /**
     * Pattern für Einträge ohne Präfix ("ISBN ")
     */
    private Pattern withoutIsbnPattern;

    private IsbnStatistics statistics;

    /**
     * Constructs ValidateIsbn instance.
     */
    public ValidateIsbn() {
        validator = ISBNValidator.getInstance();
        isbnPattern = Pattern.compile("ISBN (\\S+).?");
        withoutIsbnPattern = Pattern.compile("(\\S+).?");
    }

    /**
     * Führt die ISBN Prüfung für Satz durch.
     *
     * Diese Methode iteriert über die gefundenen MAB540 Felder und prüft jedes einzelne auf inkorrekte ISBN Einträge.
     *
     * @param satz MabSatz für Prüfung
     */
    @Override
    public void process(MabSatz satz) {
        List<IMabFeld> felder = satz.getFelder(MabFeldDefinition.MAB540);

        for (IMabFeld feld : felder) {
            checkField(satz, feld);
        }
    }

    /**
     * Prüft den Inhalt von MAB540.
     *
     * @param satz IMabSatz
     * @param feld Zu prüfendes Feld
     */
    public void checkField(IMabSatz satz, IMabFeld feld) {
        String content = feld.getInhalt();
        String indicator = Character.toString(( char )feld.getIndikator());

        boolean isbnValid = isValid(content);

        switch (indicator.charAt(0)) {
            case 'a':
                // invalid ISBN in 540a
                if (!isbnValid) {
                    // always report invalid ISBN in 540a
                    Set<IsbnErrorType.Reason> reasons = analyseInvalidIsbn(content);
                    addDatasetError(new IsbnErrorType(IsbnErrorType.INVALID_ISBN_IN_540A, reasons), satz, content);
                    statistics.addInvalidIsbnReasons(reasons);
                }
                break;
            case 'b':
                // valid ISBN in 540b
                if (isbnValid) {
                    addDatasetError(new IsbnErrorType(IsbnErrorType.VALID_ISBN_IN_540B), satz, content);
                }
                break;
            case 'z':
                // TODO Prüfe auf Inhalte, die wie ISBNs aussehen
                if (content.contains("ISBN")) {
                    addDatasetError(new IsbnErrorType(IsbnErrorType.ISBN_IN_540Z), satz, content);
                }
                break;
            default:
                // TODO Erstelle Statistik über 540_ (formal nicht geprüft)?
                break;
        }
    }

    /**
     * Prüft mögliche Ursachen für einen ungültigen ISBN Eintrag.
     * @param content
     * @return Set<IsbnErrorType.Reason>
     *
     * TODO move into separate class?
     */
    public Set<IsbnErrorType.Reason> analyseInvalidIsbn(String content) {
        Set<IsbnErrorType.Reason> reason = new HashSet<IsbnErrorType.Reason>();

        if (isIsbnMissing(content)) {
            reason.add(IsbnErrorType.Reason.ISBN_WITHOUT_LEADER);
            /* TODO check if checksum valid
            if (!validator.isValid(content)) {
            }
            */
        }
        else {
            // TODO perform same checks if "ISBN_" is missing
            // if (isContainsMultipleISBN(content)) {
                // TODO addDatasetError(new IsbnErrorType(IsbnErrorType.MULTIPLE_ISBN), satz, content);
            // }

            if (isValid(content, true)) {
                reason.add(IsbnErrorType.Reason.ISBN_BAD_FORMAT);
            }

            if (isSmallX(content)) {
                reason.add(IsbnErrorType.Reason.ISBN_SMALL_X);
            }
        }

        return reason;
    }

    /**
     *
     *
     * TODO what to do with message
     */
    protected void addDatasetError(IsbnErrorType errorType, IMabSatz satz, String content) {
        getErrorStatistics().addError(new IsbnError(satz, errorType, content));
    }

    protected boolean isIsbnMissing(String content) {
        return !content.startsWith("ISBN ");
    }

    protected boolean isSmallX(String content) {
        return isValid(content.toUpperCase());
    }

    /**
     * Prepares a ISBN for validation.
     * @param content Field content
     * @return String ISBN value
     */
    public String prepareValue(String content) {
        Matcher matcher = isbnPattern.matcher(content);
        if (matcher.find()) {
            return matcher.group(1);
        }
        else {
            return null;
        }
    }

    /**
     * Checks if a ISBN value is valid.
     * @param isbn String ISBN value
     * @return true - if ISBN is valid; false - if ISBN is not valid
     */
    public boolean isValid(final String isbn) {
        return isValid(isbn, false);
    }

    /**
     *
     * @param content Content of field
     * @return true - if field seems to contain multiple ISBNs; false - if not
     */
    public boolean isContainsMultipleISBN(String content) {
        String value = prepareValue(content);

        if (value == null) {
            return false;
        }

        int digits = 0;

        for (int pos = 0; pos < value.length(); pos++) {
            char charAt = value.charAt(pos);

            if (Character.isDigit(charAt) || charAt == 'x' || charAt == 'X') {
                digits++;
            }
        }

        return digits >= 20;
    }

    protected boolean isValid(String isbn, boolean removeSeparator) {
        String value = prepareValue(isbn);
        if (removeSeparator) {
            value = StringUtils.remove(value, "-");
        }
        LOG.debug("isValid(" + value + ")");
        return validator.isValid(value);
    }

    @Autowired
    public void setIsbnStatistics(IsbnStatistics statistics) {
        this.statistics = statistics;
    }

}
