/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.parser;

import de.ddb.charset.MabCharset;
import java.io.IOException;
import java.io.Reader;
import java.util.concurrent.CountDownLatch;
import org.apache.log4j.Logger;

/**
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class MabDatasetCounter {

    private static final Logger LOG = Logger.getLogger(MabDatasetCounter.class);

    private static final int BUFFER_SIZE = 8 * 1024;

    private Reader source;

    private int datasetCounter;

    public void init() {
        datasetCounter = 0;
    }

    public void parse() throws IOException {
        int numberOfThreads = Runtime.getRuntime().availableProcessors();

        LOG.debug("Using " + numberOfThreads + " counting threads.");

        CountDownLatch done = new CountDownLatch(numberOfThreads);

        CounterThread[] threads = new CounterThread[numberOfThreads];

        for (int i = 0; i < numberOfThreads; i++) {
            threads[i] = new CounterThread(source, done);

            threads[i].start();
        }

        try {
            done.await();
        }
        catch (InterruptedException e) {
            LOG.error("Error counting datasets.", e);
        }

        for (CounterThread counter : threads) {
            datasetCounter += counter.getResult();
        }
    }

    public void reset() {
        datasetCounter = 0;
    }

    public Reader getSource() {
        return source;
    }

    public void setSource(final Reader src) {
        this.source = src;
    }

    public int getDatasetCounter() {
        return datasetCounter;
    }

    private class CounterThread extends Thread {

        private Reader source;

        private CountDownLatch done;

        private int datasetCounter = 0;

        public CounterThread(final Reader src, final CountDownLatch latch) {
            this.source = src;
            this.done = latch;
        }

        @Override
        public void run() {
            char[] buf = new char[BUFFER_SIZE];

            int charsRead = 0;

            try {
                while ((charsRead = source.read(buf, 0, buf.length)) != -1) {
                    for (int i = 0; i < charsRead; i++) {
                      if (buf[i] == MabCharset.SATZENDEZEICHEN) {
                          datasetCounter++;
                      }
                    }
                }

                done.countDown();
            }
            catch (IOException e) {
                System.err.println(e);
            }
        }

        public int getResult() {
            return datasetCounter;
        }

    }

}
