/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.extension;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Stores dataset IDs for checking references.
 *
 * The implementation stores all added dataset identifiers (MAB001) by dataset
 * type for checking references.
 *
 * If a reference check fails the missing identifier is added to a list. When
 * a new identifier is added that list is checked and if the new identifier has
 * been previously missing it will be removed from the list of missing IDs.
 *
 * After the processing is done the list of missing identifiers contains the
 * references that were not found within the data.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class DefaultReferencesStore implements ReferencesStore {

    /**
     * Holds MAB001 content for all MAB datasets by dataset type.
     */
    private Map<String, Set<String>> satzIdsByType =
            new HashMap<String, Set<String>>();

    /**
     * Holds MAB001 for datasets that are referenced and have not been found
     * yet.
     */
    private Map<String, Set<String>> missingIdsByType =
            new HashMap<String, Set<String>>();

    /**
     * Set of listeners.
     */
    private Set<ReferencesStoreListener> listeners =
            new HashSet<ReferencesStoreListener>();

    /**
     * ID of dataset last added.
     *
     * TODO refactor? needed?
     */
    private String currentDatasetId;

    /**
     * Key for missing datasets without type.
     */
    private static final String NO_TYPE = "NO_TYPE";

    /**
     * Adds a dataset ID for reference.
     *
     * @param datasetId Dataset ID (MAB001)
     * @param satzTyp Type of dataset (e.g. 'h' or 'u')
     */
    @Override
    public void addDataset(final String datasetId, final char satzTyp)
            throws IdentifierAlreadyExistsException {
        String datasetType = Character.toString(satzTyp);

        this.currentDatasetId = datasetId;

        if (isIdentifierExists(datasetId)) {
            throw new IdentifierAlreadyExistsException(datasetId);
        }

        Set<String> satzIdsForType = satzIdsByType.get(datasetType);

        if (satzIdsForType == null) {
            satzIdsForType = new HashSet<String>();
            satzIdsByType.put(datasetType, satzIdsForType);
        }

        satzIdsForType.add(datasetId);

        Set<String> missingIdsForNoType = missingIdsByType.get(NO_TYPE);
        Set<String> missingIdsForType = missingIdsByType.get(datasetType);

        if ((missingIdsForType != null)
                && missingIdsForType.remove(datasetId)) {
            fireDatasetFound(datasetId, datasetType);
        }
        else if ((missingIdsForNoType != null)
                && missingIdsForNoType.remove(datasetId)) {
            fireDatasetFound(datasetId, datasetType);
        }
    }

    private boolean isIdentifierExists(String identifier) {
        for (Map.Entry<String, Set<String>> entry
                : satzIdsByType.entrySet()) {
            if (entry.getValue().contains(identifier)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Adds a listener for the references store.
     * @param listener ReferencesStoreListener
     */
    @Override
    public void addListener(final ReferencesStoreListener listener) {
        listeners.add(listener);
    }

    /**
     * Returns set of dataset identifiers that were not found.
     * @return Set<String> dataset identifiers (MAB001)
     */
    @Override
    public Set<String> getMissingDatasets() {
        Set<String> allMissingIds = new HashSet<String>();

        for (Set<String> missingIds : missingIdsByType.values()) {
            allMissingIds.addAll(missingIds);
        }

        return allMissingIds;
    }

    /**
     * Checks if a dataset identifier exists in references store.
     * @param datasetId String dataset identifier (MAB001)
     * @return true - if identifier is found, false - if identifier is not found
     */
    @Override
    public boolean isDatasetExists(final String datasetId) {
        return isDatasetExists(datasetId, null);
    }

    /**
     * Checks if a dataset identifier exists in references store.
     * @param datasetId String dataset identifier (MAB001)
     * @param datasetType String dataset type
     * @return true - if identifier is found, false - if identifier not found
     */
    @Override
    public boolean isDatasetExists(final String datasetId,
            final String datasetType) {
        if (datasetType == null) {
            boolean found = false;

            for (Map.Entry<String, Set<String>> entry
                    : satzIdsByType.entrySet()) {
                if (entry.getValue().contains(datasetId)) {
                    found = true;
                    break;
                }
            }

            if (!found) {
                addMissingDataset(datasetId, null);
            }

            return found;
        }
        else {
            Set<String> satzIdsForType = satzIdsByType.get(datasetType);

            if (satzIdsForType != null && satzIdsForType.contains(datasetId)) {
                return true;
            }
            else {
                addMissingDataset(datasetId, datasetType);
                return false;
            }
        }
    }

    /**
     * Removes a ReferencesStoreListener.
     * @param listener ReferencesStoreListener
     */
    @Override
    public void removeListener(final ReferencesStoreListener listener) {
        listeners.remove(listener);
    }

    /**
     * Getter for dataset ID that was last added.
     * @return String MAB001 value
     */
    @Override
    public String getCurrentDataset() {
        return currentDatasetId;
    }

    /**
     * Calls datasetFound for all registered listeners.
     * @param datasetId String dataset identifier
     * @param datasetType String dataset type (e.g. 'h' or 'u')
     */
    private void fireDatasetFound(final String datasetId,
            final String datasetType) {
        for (ReferencesStoreListener listener : listeners) {
            listener.datasetFound(datasetId, datasetType);
        }
    }

    /**
     * Adds a dataset ID that is not contained in stored references.
     * @param datasetId String dataset identifier (MAB001)
     * @param datasetType String type of dataset
     */
    private void addMissingDataset(final String datasetId,
            final String datasetType) {
        String type;

        if (datasetType == null) {
            type = NO_TYPE;
        }
        else {
            type = datasetType;
        }

        Set<String> missingIdsForType = missingIdsByType.get(type);

        if (missingIdsForType == null) {
            missingIdsForType = new HashSet<String>();
            missingIdsByType.put(type, missingIdsForType);
        }

        missingIdsForType.add(datasetId);
    }

}
