/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.util.*;

import de.ddb.charset.MabCharset;

/**
 * Class representing a MAB dataset.
 *
 * MabFeld Objekte werden direkt gespeichert und auch wieder zurück geliefert. Das heißt man kann den Inhalt eines
 * Feldes modifizieren, z.B. MAB001, und die Daten des MabSatz Objektes werden direkt beeinflußt. Man muß also darauf
 * achten, das man MabFeld Objekte nicht zu mehreren Sätzen hinzufügt, da sonst bei Änderungen mehrere Sätze
 * gleichzeitig modifiziert werden.
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO add exceptions to model (bad Kennung, ...)
 *
 * TODO keep all felder in a sorted set/list
 *
 * TODO Indikator als char ist unhandlich, da der Wert nicht NULL gesetzt werden kann (momentan wird 0 verwendet)
 */
public final class MabSatz implements IMabSatz {

    /**
     * Kennung fuer MAB Satz.
     */
    private String kennung;

    /**
     * List of MAB fields for dataset.
     */
    private List<IMabFeld> felder = null;

    /**
     *
     */
    private Map<String, List<IMabFeld>> felderByCategory = null;

    /**
     * Creates MAB dataset with header.
     * @param header Satzkennung
     */
    public MabSatz(final String header) {
        this.kennung = header;
    }

    /**
     * Creates MAB dataset with header and fields.
     * @param header Header of dataset
     * @param fields Fields for MAB dataset
     */
    public MabSatz(final String header, final List<IMabFeld> fields) {
        this.kennung = header;
        this.felder = fields;
    }

    public MabSatz(final MabSatzkennung satzkennung) {
        this.kennung = satzkennung.getValue();
    }

    @Override
    public MabSatzkennung getSatzkennung() {
        return new MabSatzkennung(this.kennung);
    }

    @Override
    public String getId() {
        List<IMabFeld> felder = getFelder(MabFeldDefinition.MAB001);

        if (felder != null && felder.size() > 0) {
            return felder.get(0).getInhalt();
        }
        else {
            return null;
        }
    }

    /**
     * Adds a MAB field to the dataset.
     * @param feld MAB field
     */
    @Override
    public void addFeld(final IMabFeld feld) {
        internAddFeld(feld);
    }

    @Override
    public void removeFeld(IMabFeld feld) {
        internRemoveFeld(feld);
    }

    public Set<String> getKategorien() {
        if (felderByCategory != null) {
            return felderByCategory.keySet();
        }
        else {
            return null;
        }
    }

    /**
     * Getter for subfields.
     * @return List of MabFeld objects
     */
    @Override
    public List<IMabFeld> getFelder() {
        return felder;
    }

    @Override
    public List<IMabFeld> getFelder(String category) {
        List<IMabFeld> felder = new ArrayList<IMabFeld>();

        if (felderByCategory != null) {
            for (String key : felderByCategory.keySet()) {
                if (key.startsWith(category)) {
                    felder.addAll(felderByCategory.get(key));
                }
            }
        }

        return felder;
    }

    @Override
    public List<IMabFeld> getFelder(String category, char indikator) {
        List<IMabFeld> felder = null;

        if (felderByCategory != null) {
            if (indikator != 0) {
                felder = felderByCategory.get(category + indikator);
            }
            else {
                felder = getFelder(category);
            }
        }

        return felder;
    }

    @Override
    public List<IMabFeld> getFelder(MabFeldDefinition feldDef) {
        List<IMabFeld> felder = new ArrayList<IMabFeld>();

        String feldnummer = feldDef.getFeldnummerString();

        return getFelder(feldnummer);
    }

    /**
     * Getter for dataset header.
     * @return Header as string
     */
    public String getKennung() {
        return kennung;
    }

    @Override
    public boolean isFeldPresent(String kategorie) {
        return getFelder(kategorie).size() > 0;
    }

    @Override
    public boolean isFeldPresent(final String category, final char indikator) {
        if (indikator != 0) {
            String name = category + indikator;

            return felderByCategory != null && felderByCategory.keySet().contains(name);
        }
        else {
            return isFeldPresent(category);
        }
    }

    /**
     * Converts MAB dataset to string.
     * @return Entire MAB dataset as string.
     */
    @Override
    public String toString() {
        String satz = kennung;
        if (felder != null) {
            for (IMabFeld aFelder : felder) {
                satz += (aFelder).toString();
            }
        }
        satz += Character.toString(MabCharset.SATZENDEZEICHEN);
        return satz;
    }

    /**
     * Fügt ein Feld zur Liste aller Felder und zu einem Hash nach Feldnummer hinzu.
     *
     * @param feld IMabFeld
     */
    private void internAddFeld(IMabFeld feld) {
        if (this.felder == null) {
            this.felder = new LinkedList<IMabFeld>();
        }
        this.felder.add(feld);

        if (this.felderByCategory == null) {
            this.felderByCategory = new HashMap<String, List<IMabFeld>>();
        }

        String bezeichner = feld.getBezeichnung();

        List<IMabFeld> felderForCategory = this.felderByCategory.get(bezeichner);

        if (felderForCategory == null) {
            felderForCategory = new ArrayList<IMabFeld>();
            this.felderByCategory.put(bezeichner, felderForCategory);
        }

        felderForCategory.add(feld);
    }

    /**
     * Entfernt ein Feld von den internen Listen und Hashes.
     *
     * @param feld IMabFeld
     *
     * TODO exception handling (feld not present)?
     */
    private void internRemoveFeld(IMabFeld feld) {
        if (this.felder == null) {
            return;
        }

        this.felder.remove(feld);

        String bezeichner = feld.getBezeichnung();

        List<IMabFeld> felderForCategory = this.felderByCategory.get(bezeichner);

        if (felderForCategory != null) {
            felderForCategory.remove(feld);
        }
    }

}
