/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.net.URL;
import java.util.*;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.configuration.tree.xpath.XPathExpressionEngine;

/**
 * Speichert die m&ouml;glichen Belegungen eines Feldes.
 *
 * Also alle zugelassenen Indikatoren,sowie die Unterfeldkennzeichen. Beinhaltet Methoden um Feldnummern in Strings
 * umzuwandeln, die Indikatoren und Unterfeldkennzeichen auf ihre Zulässigkeit zum Feld zu überprüfen und das gesamte
 * Feld in einen String zu verwandeln.
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public enum MabFeldDefinition {

    // Im Folgenden werden nun alle definierten Mab2-Titel-Felder
    // in die HashMap aufgenommen. Die folgende Version ist auf
    // dem Stand der Formaterweiterungen und -aenderungen vom
    // Juni 2006.
    // Quellen:
    //  - http://www.d-nb.de/standardisierung/txt/titelmab.txt
    //  - http://www.d-nb.de/standardisierung/txt/erw-mab.txt
    //  - http://www.d-nb.de/standardisierung/txt/segm000.txt

    // CHECKSTYLE:OFF
    MAB001(1, new char[]{' '}),
    MAB002(2, new char[]{'a', 'b'}, new char[]{}),
    MAB003(3, new char[]{' '}),
    MAB004(4, new char[]{' '}),
    MAB005(5, new char[]{'n', 'v'}),
    MAB006(6, new char[]{'n', 'v'}),
    MAB010(10, new char[]{' '}),
    MAB011(11, new char[]{' '}),
    MAB012(12, new char[]{' '}),
    MAB015(15, new char[]{' '}),
    MAB016(16, new char[]{' '}),
    MAB020(20, new char[]{' ', 'a', 'b', 'c'}),
    MAB021(21, new char[]{' ', 'a', 'b', 'c'}),
    MAB022(22, new char[]{' ', 'a', 'b', 'c'}),
    MAB023(23, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'f'}, new char[]{'a'}),
    MAB025(25, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'l', 'o',
        'z'}),
    MAB026(26, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i'}),
    MAB027(27, new char[]{' ', 'a', 'b'}),
    MAB028(28, new char[]{' ', 'a', 'b', 'c'}),
    MAB029(29, new char[]{' '}),

    // 030 - 035 Allgemeine verarbeitungstechnische Angaben
    MAB030(30, new char[]{' '}),
    MAB031(31, new char[]{' '}, new char[]{'a', 'b', 'c', 'd', 'e'}),

    // 036 - 049 Allgemeine codierte Angaben
    MAB036(36, new char[]{' ', 'a', 'b', 'c', 'z'}),
    MAB037(37, new char[]{' ', 'a', 'b', 'c', 'z'}),
    MAB038(38, new char[]{' ', 'a', 'b', 'c', 'z'}),
    MAB039(39, new char[]{' ', 'a', 'b', 'c', 'z'}),
    MAB040(40, new char[]{' ', 'a', 'b'}),
    MAB041(41, new char[]{' '}),

    // 050 - 064 Veroeffentlichungs- und materialspezifische Angaben
    MAB050(50, new char[]{' '}),
    MAB051(51, new char[]{' '}),
    MAB052(52, new char[]{' '}),
    MAB053(53, new char[]{' '}),
    MAB057(57, new char[]{' '}),
    MAB058(58, new char[]{' '}),

    // 065 - 069 Normdatenspezifische Angaben
    MAB065(65, new char[]{' '}),
    MAB066(66, new char[]{' '}),
    MAB067(67, new char[]{' '}),
    MAB068(68, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'z'}),

    // 070 - 075 Anwenderspezifische Daten und Codes
    MAB070(70, new char[]{' ', 'a', 'b'}),
    MAB071(71, new char[]{' ', 'a', 'b', 'c', 'd'}),
    MAB072(72, new char[]{' '}),
    MAB073(73, new char[]{' '}),
    MAB074(74, new char[]{' '}),
    MAB075(75, new char[]{' '}),

    // Anwenderspezifische Angaben
    MAB076(76),
    MAB077(77),
    MAB078(78),
    MAB079(79),
    MAB080(80),
    MAB081(81),
    MAB082(82),
    MAB083(83),
    MAB084(84),
    MAB085(85),
    MAB086(86),
    MAB087(87),
    MAB088(88),

    // Bandangaben
    MAB089(89, new char[]{' '}),
    MAB090(90, new char[]{' '}),

    // SEGMENT 1--
    // Personennamen
    MAB100(100, new char[]{' ', 'b', 'c', 'f', 'e'}),
    MAB101(101, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB102(102, new char[]{' ', 'a', 'b', 'c'}),
    MAB103(103, new char[]{' '}),

    MAB104(104, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB105(105, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB106(106, new char[]{' ', 'a', 'b', 'c'}),
    MAB107(107, new char[]{' '}),

    MAB108(108, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB109(109, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB110(110, new char[]{' ', 'a', 'b', 'c'}),
    MAB111(111, new char[]{' '}),

    MAB112(112, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB113(113, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB114(114, new char[]{' ', 'a', 'b', 'c'}),
    MAB115(115, new char[]{' '}),

    MAB116(116, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB117(117, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB118(118, new char[]{' ', 'a', 'b', 'c'}),
    MAB119(119, new char[]{' '}),

    MAB120(120, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB121(121, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB122(122, new char[]{' ', 'a', 'b', 'c'}),
    MAB123(123, new char[]{' '}),

    MAB124(124, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB125(125, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB126(126, new char[]{' ', 'a', 'b', 'c'}),
    MAB127(127, new char[]{' '}),

    MAB128(128, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB129(129, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB130(130, new char[]{' ', 'a', 'b', 'c'}),
    MAB131(131, new char[]{' '}),

    MAB132(132, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB133(133, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB134(134, new char[]{' ', 'a', 'b', 'c'}),
    MAB135(135, new char[]{' '}),

    MAB136(136, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB137(137, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB138(138, new char[]{' ', 'a', 'b', 'c'}),
    MAB139(139, new char[]{' '}),

    MAB140(140, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB141(141, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB142(142, new char[]{' ', 'a', 'b', 'c'}),
    MAB143(143, new char[]{' '}),

    MAB144(144, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB145(145, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB146(146, new char[]{' ', 'a', 'b', 'c'}),
    MAB147(147, new char[]{' '}),

    MAB148(148, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB149(149, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB150(150, new char[]{' ', 'a', 'b', 'c'}),
    MAB151(151, new char[]{' '}),

    MAB152(152, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB153(153, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB154(154, new char[]{' ', 'a', 'b', 'c'}),
    MAB155(155, new char[]{' '}),

    MAB156(156, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB157(157, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB158(158, new char[]{' ', 'a', 'b', 'c'}),
    MAB159(159, new char[]{' '}),

    MAB160(160, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB161(161, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB162(162, new char[]{' ', 'a', 'b', 'c'}),
    MAB163(163, new char[]{' '}),

    MAB164(164, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB165(165, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB166(166, new char[]{' ', 'a', 'b', 'c'}),
    MAB167(167, new char[]{' '}),

    MAB168(168, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB169(169, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB170(170, new char[]{' ', 'a', 'b', 'c'}),
    MAB171(171, new char[]{' '}),

    MAB172(172, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB173(173, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB174(174, new char[]{' ', 'a', 'b', 'c'}),
    MAB175(175, new char[]{' '}),

    MAB176(176, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB177(177, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB178(178, new char[]{' ', 'a', 'b', 'c'}),
    MAB179(179, new char[]{' '}),

    MAB180(180, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB181(181, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB182(182, new char[]{' ', 'a', 'b', 'c'}),
    MAB183(183, new char[]{' '}),

    MAB184(184, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB185(185, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB186(186, new char[]{' ', 'a', 'b', 'c'}),
    MAB187(187, new char[]{' '}),



    MAB188(188, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB189(189, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB190(190, new char[]{' ', 'a', 'b', 'c'}),
    MAB191(191, new char[]{' '}),

    MAB192(192, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB193(193, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB194(194, new char[]{' ', 'a', 'b', 'c'}),
    MAB195(195, new char[]{' '}),

    MAB196(196, new char[]{'a', 'b', 'c', 'f', 'e'}),
    MAB197(197, new char[]{' ', 'a', 'b', 'c', 'd', 'z'}),
    MAB198(198, new char[]{' ', 'a', 'b', 'c'}),
    MAB199(199, new char[]{' '}),

    // SEGEMENT 2--
    // Koerperschaftsnamen
    MAB200(200, new char[]{' ', 'b', 'c', 'e'}),
    MAB201(201, new char[]{' ', 'z'}),
    MAB202(202, new char[]{' ', 'a', 'b', 'c'}),

    MAB204(204, new char[]{'a', 'b', 'c', 'e'}),
    MAB205(205, new char[]{' ', 'z'}),
    MAB206(206, new char[]{' ', 'a', 'b', 'c'}),

    MAB208(208, new char[]{'a', 'b', 'c', 'e'}),
    MAB209(209, new char[]{' ', 'z'}),
    MAB210(210, new char[]{' ', 'a', 'b', 'c'}),

    MAB212(212, new char[]{'a', 'b', 'c', 'e'}),
    MAB213(213, new char[]{' ', 'z'}),
    MAB214(214, new char[]{' ', 'a', 'b', 'c'}),

    MAB216(216, new char[]{'a', 'b', 'c', 'e'}),
    MAB217(217, new char[]{' ', 'z'}),
    MAB218(218, new char[]{' ', 'a', 'b', 'c'}),

    MAB220(220, new char[]{'a', 'b', 'c', 'e'}),
    MAB221(221, new char[]{' ', 'z'}),
    MAB222(222, new char[]{' ', 'a', 'b', 'c'}),

    MAB224(224, new char[]{'a', 'b', 'c', 'e'}),
    MAB225(225, new char[]{' ', 'z'}),
    MAB226(226, new char[]{' ', 'a', 'b', 'c'}),

    MAB228(228, new char[]{'a', 'b', 'c', 'e'}),
    MAB229(229, new char[]{' ', 'z'}),
    MAB230(230, new char[]{' ', 'a', 'b', 'c'}),

    MAB232(232, new char[]{'a', 'b', 'c', 'e'}),
    MAB233(233, new char[]{' ', 'z'}),
    MAB234(234, new char[]{' ', 'a', 'b', 'c'}),

    MAB236(236, new char[]{'a', 'b', 'c', 'e'}),
    MAB237(237, new char[]{' ', 'z'}),
    MAB238(238, new char[]{' ', 'a', 'b', 'c'}),

    MAB240(240, new char[]{'a', 'b', 'c', 'e'}),
    MAB241(241, new char[]{' ', 'z'}),
    MAB242(242, new char[]{' ', 'a', 'b', 'c'}),

    MAB244(244, new char[]{'a', 'b', 'c', 'e'}),
    MAB245(245, new char[]{' ', 'z'}),
    MAB246(246, new char[]{' ', 'a', 'b', 'c'}),

    MAB248(248, new char[]{'a', 'b', 'c', 'e'}),
    MAB249(249, new char[]{' ', 'z'}),
    MAB250(250, new char[]{' ', 'a', 'b', 'c'}),

    MAB252(252, new char[]{'a', 'b', 'c', 'e'}),
    MAB253(253, new char[]{' ', 'z'}),
    MAB254(254, new char[]{' ', 'a', 'b', 'c'}),

    MAB256(256, new char[]{'a', 'b', 'c', 'e'}),
    MAB257(257, new char[]{' ', 'z'}),
    MAB258(258, new char[]{' ', 'a', 'b', 'c'}),

    MAB260(260, new char[]{'a', 'b', 'c', 'e'}),
    MAB261(261, new char[]{' ', 'z'}),
    MAB262(262, new char[]{' ', 'a', 'b', 'c'}),

    MAB264(264, new char[]{'a', 'b', 'c', 'e'}),
    MAB265(265, new char[]{' ', 'z'}),
    MAB266(266, new char[]{' ', 'a', 'b', 'c'}),

    MAB268(268, new char[]{'a', 'b', 'c', 'e'}),
    MAB269(269, new char[]{' ', 'z'}),
    MAB270(270, new char[]{' ', 'a', 'b', 'c'}),

    MAB272(272, new char[]{'a', 'b', 'c', 'e'}),
    MAB273(273, new char[]{' ', 'z'}),
    MAB274(274, new char[]{' ', 'a', 'b', 'c'}),

    MAB276(276, new char[]{'a', 'b', 'c', 'e'}),
    MAB277(277, new char[]{' ', 'z'}),
    MAB278(278, new char[]{' ', 'a', 'b', 'c'}),

    MAB280(280, new char[]{'a', 'b', 'c', 'e'}),
    MAB281(281, new char[]{' ', 'z'}),
    MAB282(282, new char[]{' ', 'a', 'b', 'c'}),

    MAB284(284, new char[]{'a', 'b', 'c', 'e'}),
    MAB285(285, new char[]{' ', 'z'}),
    MAB286(286, new char[]{' ', 'a', 'b', 'c'}),

    MAB288(288, new char[]{'a', 'b', 'c', 'e'}),
    MAB289(289, new char[]{' ', 'z'}),
    MAB290(290, new char[]{' ', 'a', 'b', 'c'}),

    MAB292(292, new char[]{'a', 'b', 'c', 'e'}),
    MAB293(293, new char[]{' ', 'z'}),
    MAB294(294, new char[]{' ', 'a', 'b', 'c'}),

    MAB296(296, new char[]{'a', 'b', 'c', 'e'}),
    MAB297(297, new char[]{' ', 'z'}),
    MAB298(298, new char[]{' ', 'a', 'b', 'c'}),

    // SEGMENT 3--
    // Sachtitel
    MAB300(300, new char[]{' '}),
    MAB304(304, new char[]{' ', 'a', 'b'}),
    MAB305(305, new char[]{' ', 'a', 'b', 'c'}),
    MAB310(310, new char[]{' ', 'a', 'b'}),
    MAB331(331, new char[]{' ', 'a', 'b'}), // Haupttitel
    MAB333(333, new char[]{' '}),
    MAB334(334, new char[]{' '}),
    MAB335(335, new char[]{' ', 'a', 'b'}),
    MAB340(340, new char[]{' ', 'a', 'b'}),
    MAB341(341, new char[]{' ', 'a', 'b'}),
    MAB342(342, new char[]{' '}),
    MAB343(343, new char[]{' ', 'a', 'b'}),
    MAB344(344, new char[]{' ', 'a', 'b'}),
    MAB345(345, new char[]{' ', 'a', 'b'}),
    MAB346(346, new char[]{' '}),
    MAB347(347, new char[]{' ', 'a', 'b'}),
    MAB348(344, new char[]{' ', 'a', 'b'}),
    MAB349(345, new char[]{' ', 'a', 'b'}),
    MAB350(346, new char[]{' '}),
    MAB351(347, new char[]{' ', 'a', 'b'}),
    MAB352(344, new char[]{' ', 'a', 'b'}),
    MAB353(345, new char[]{' ', 'a', 'b'}),
    MAB354(346, new char[]{' '}),
    MAB355(347, new char[]{' ', 'a', 'b'}),
    MAB359(359, new char[]{' '}),
    MAB360(360, new char[]{' '}),
    MAB361(361, new char[]{' '}),
    MAB365(365, new char[]{' '}),
    MAB369(369, new char[]{' '}),
    MAB370(370, new char[]{'a', 'b', 'c'}),
    MAB376(376, new char[]{' ', 'a', 'b', 'c', 'd'}),

    // SEGMENT 4--
    // 400 - 437 Veroeffentlichungsvermerk, Umfang, Beigaben
    MAB400(400, new char[]{' '}),
    MAB403(403, new char[]{' '}),
    MAB405(405, new char[]{' '}),
    MAB406(406, new char[]{' ', 'a', 'b'}, new char[]{'5', 'f', 'd', 'e', 'b',
        'c', 'j', 'h', 'g', 'n', 'o', 'l', 'm', 'k', 'i', 'q'}),
    MAB407(407, new char[]{' '}),
    MAB410(410, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB411(411, new char[]{' '}),
    MAB412(412, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB415(415, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB416(416, new char[]{' '}),
    MAB417(417, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB418(418, new char[]{' ', 'a', 'b', 'c', 'u'}, new char[]{'a', 'b', 'c',
        'd', 'e', 'f', 'g', 'h', 'u'}),
    MAB420(420, new char[]{' '}),
    MAB425(425, new char[]{' ', 'a', 'b', 'c', 'p'}),
    MAB426(426, new char[]{' ', 'a'}),
    MAB427(427, new char[]{' '}),
    MAB429(429, new char[]{' '}),
    MAB431(431, new char[]{' '}),
    MAB432(432, new char[]{' '}),
    MAB433(433, new char[]{' ', 'a', 'b', 'c'}),
    MAB434(434, new char[]{' ', 'a', 'b', 'c', 'd'}),
    MAB435(435, new char[]{' ', 'a'}),
    MAB437(437, new char[]{' '}),

    // 451 - 496 Gesamttitelangaben
    MAB451(451, new char[]{' ', 'a', 'b'}),
    MAB452(452, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB453(453, new char[]{' ', 'm', 'r'}),
    MAB454(454, new char[]{' ', 'a', 'b', 'c'}),
    MAB455(455, new char[]{' '}),
    MAB456(456, new char[]{' '}),

    MAB461(461, new char[]{' ', 'a', 'b', 'c'}),
    MAB462(462, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB463(463, new char[]{' ', 'm', 'r'}),
    MAB464(464, new char[]{' ', 'a', 'b', 'c'}),
    MAB465(465, new char[]{' '}),
    MAB466(466, new char[]{' '}),
    MAB471(471, new char[]{' ', 'a', 'b', 'c'}),
    MAB472(472, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB473(473, new char[]{' ', 'm', 'r'}),
    MAB474(474, new char[]{' ', 'a', 'b', 'c'}),
    MAB475(475, new char[]{' '}),
    MAB476(476, new char[]{' '}),
    MAB481(481, new char[]{' ', 'a', 'b', 'c'}),
    MAB482(482, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB483(483, new char[]{' ', 'm', 'r'}),
    MAB484(484, new char[]{' ', 'a', 'b', 'c'}),
    MAB485(485, new char[]{' '}),
    MAB486(486, new char[]{' '}),
    MAB491(491, new char[]{' ', 'a', 'b', 'c'}),
    MAB492(492, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB493(493, new char[]{' ', 'm', 'r'}),
    MAB494(494, new char[]{' ', 'a', 'b', 'c'}),
    MAB495(495, new char[]{' '}),
    MAB496(496, new char[]{' '}),

    // SEGEMENT 5--
    // 501 - 539 Fussnoten
    MAB501(501, new char[]{' '}),
    MAB502(502, new char[]{' '}),
    MAB503(503, new char[]{' ', 'a'}),
    MAB504(504, new char[]{' '}),
    MAB505(505, new char[]{' '}),
    MAB507(507, new char[]{' '}),
    MAB508(508, new char[]{' '}),
    MAB509(509, new char[]{' '}),
    MAB510(510, new char[]{' '}),
    MAB511(511, new char[]{' '}),
    MAB512(512, new char[]{' ', 'a'}),
    MAB513(513, new char[]{' '}),
    MAB515(515, new char[]{' '}),
    MAB516(516, new char[]{' ', 'a', 'b', 'c', 'd'}),
    MAB517(517, new char[]{' ', 'a', 'b', 'c'}),
    MAB518(518, new char[]{'a', 'b'}),
    MAB519(519, new char[]{' '}),
    MAB522(522, new char[]{' '}),
    MAB523(523, new char[]{' '}),
    MAB524(524, new char[]{' '}),
    MAB525(525, new char[]{' '}),
    MAB526(526, MabIndicator.BLANK_X_Y_Z), // new char[]{' ', 'x', 'y', 'z'}
    MAB527(527, MabIndicator.BLANK_X_Y_Z),
    MAB528(528, MabIndicator.BLANK_X_Y_Z),
    MAB529(529, MabIndicator.BLANK_X_Y_Z),
    MAB530(530, MabIndicator.BLANK_X_Y_Z),
    MAB531(531, MabIndicator.BLANK_X_Y_Z),
    MAB532(532, MabIndicator.BLANK_X_Y_Z),
    MAB533(533, MabIndicator.BLANK_X_Y_Z),
    MAB534(534, MabIndicator.BLANK_X_Y_Z),
    MAB535(535, new char[]{' '}),
    MAB536(536, new char[]{' '}),
    MAB537(537, new char[]{' '}),
    MAB538(538, new char[]{' '}),

    // 540 - 589 Standardnummern
    MAB540(540, new char[]{' ', 'a', 'b', 'z'}),
    MAB541(541, new char[]{' ', 'a', 'b', 'z'}),
    MAB542(542, new char[]{' ', 'a', 'b', 'z'}),
    MAB543(543, new char[]{' ', 'a', 'b', 'z'}),
    MAB544(544, new char[]{' ', 'a', 'b'}),
    MAB545(545, new char[]{' ', 'a', 'b'},
        new char[]{'a', 'b', 'c', 'd'}),
    MAB546(546, new char[]{' '}),
    MAB550(550, new char[]{' '}),
    MAB551(551, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'f'}),
    MAB552(552, new char[]{' ', 'a', 'b', 'c'}),
    MAB553(553, new char[]{' ', 'a', 'b'}),
    MAB554(554, new char[]{' '}),
    MAB556(556, new char[]{' ', 'a', 'b', 'c'}),
    MAB562(562, new char[]{' ', 'a', 'b', 'c'}),
    MAB564(564, new char[]{' '}),
    MAB566(566, new char[]{' '}),
    MAB568(568, new char[]{' '}),
    MAB570(570, new char[]{' '}),
    MAB574(574, new char[]{' '}),
    MAB576(576, new char[]{' '}),
    MAB578(578, new char[]{' '}),
    MAB580(580, new char[]{' '}),
    // 581-589 explizit als nicht genutzt markiert

    // 590 - 599 Herkunft (Quelle) unselbststaendig erschienener Werke
    MAB590(590, new char[]{' '}),
    MAB591(591, new char[]{' '}),
    MAB592(592, new char[]{' '}),
    MAB593(593, new char[]{' '}),
    MAB594(594, new char[]{' ', 'a', 'b', 'c'}),
    MAB595(595, new char[]{' '}),
    MAB596(596, new char[]{' ', 'a', 'b', 'c'}),
    MAB597(597, new char[]{' '}),
    MAB598(598, new char[]{' '}),
    MAB599(599, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 's'}),

    // SEGMENT 6--


    // 600 - 603 Pauschalverweisungen und Siehe-Auch-Hinweise
    MAB600(600, new char[]{'a', 'b', 'c', 'd', 'e', 'f'}),
    MAB601(601, new char[]{' '}),
    MAB602(602, new char[]{'a', 'b'}),
    MAB603(603, new char[]{' '}),

    // 610 - 650 Ausgabevermerk Sekundarformen
    MAB610(610, new char[]{' ', 'a'}),
    MAB611(611, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB612(612, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB613(613, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB614(614, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB615(615, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB616(616, new char[]{' ', 'a', 'b', 'c', 'u'}),
    MAB617(617, new char[]{' ', 'a'}),
    MAB618(618, new char[]{' ', 'a'}),
    MAB619(619, new char[]{' ', 'a', 'b', 'c'}),
    MAB620(620, new char[]{' ', 'a', 'b'}),
    MAB621(621, new char[]{' ', 'a', 'b'}),
    MAB622(622, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB623(623, new char[]{' ', 'm', 'r'}),
    MAB624(624, new char[]{' ', 'a', 'b', 'c'}),
    MAB625(625, new char[]{' '}),
    MAB626(626, new char[]{' '}),
    MAB627(627, new char[]{' ', 'a', 'b', 'c'}),
    MAB628(628, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'z'}),
    MAB629(629, new char[]{' ', 'm', 'r'}),
    MAB630(630, new char[]{' ', 'a', 'b', 'c'}),
    MAB631(631, new char[]{' '}),
    MAB632(632, new char[]{' '}),
    MAB633(633, new char[]{' '}),
    MAB634(634, new char[]{' ', 'a', 'b', 'z'}),
    MAB635(635, new char[]{' ', 'a', 'b', 'z'}),
    MAB636(636, new char[]{' '}),
    MAB637(637, new char[]{' '}),
    MAB638(638, new char[]{' '}),
    MAB640(640, new char[]{' '}),
    MAB642(642, new char[]{' '}),
    MAB644(644, new char[]{' ', 'a'}),
    MAB645(645, new char[]{' '}),
    MAB646(646, new char[]{' ', 'a', 'b'}),
    MAB647(647, new char[]{' ', 'a', 'b'}),

    // 651 - 659 Ausgabevermerk Computerdateien
    MAB651(651, new char[]{' '},
        new char[]{'a', 'b'}),
    MAB652(652, new char[]{' ', 'a', 'b'},
        new char[]{'a', 'b'}),
    MAB653(653, new char[]{' '},
        new char[]{'a', 'b', 'c', 'd', 'e'}),
    MAB654(654, new char[]{' '},
        new char[]{'a', 'b', 'c', 'd'}),
    MAB655(655, new char[]{' ', 'a', 'b', 'c', 'd', 'e', 'h'},
        /* unterMAB g in Erweiterungslieferung gestrichen
         * es gibt einen Unterschied zwichen der Online-Kurzreferenz und dem
         * Mab-Ordner:
         * Mab-Ordner: UnterMAB s = 'groesse der Datei', UnterMAB t =
         * 'unterstuetze Terminal-Emulationen'
         * Online-Kurzreferenz UnterMAB s existiert nicht, UnterMAB t enthaelt
         * die Größe der Datei.
         * => wir erlauben s und t */
        new char[]{'a', 'b', 'c', 'd', 'f', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
            'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'x', 'z',
            '2', '3', 'A'}),

    MAB659(659, new char[]{' '},
        new char[]{'a', 'b'}),

    // 661 - 669 Ausgabevermerk Nachlaesse und Autographen
    MAB661(661, new char[]{' ', 'a', 'b', 'c', 'd'}),
    MAB662(662, new char[]{' ', 'a', 'b', 'c', 'd', 'e'}),
    MAB663(663, new char[]{' ', 'a', 'b', 'c'}),
    MAB664(664, new char[]{' ', 'a', 'b', 'c'}),
    MAB669(669, new char[]{' ', 'a', 'b'}),

    // 670 - 675 Zusaetzliche Suchkriterien
    MAB670(670, new char[]{' '}),
    MAB671(671, new char[]{' '}), // FIXME: Nachfragen bzgl. erweiterung
    MAB672(672, new char[]{' '}),
    MAB673(673, new char[]{' ', 'a', 'b', 'c'}),
    MAB674(674, new char[]{' ', 'a'}),
    MAB675(675, new char[]{' ', 'a', 'b', 'c', 'd'}),

    // 680 - 689
    MAB680(680, new char[]{' '},
        new char[]{'a', 'b'}),
    MAB681(681, new char[]{' '},
        new char[]{'a', 'b', 'c'}),
    MAB682(682, new char[]{' '}, new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm'}),
    MAB683(683, new char[]{' '},
        new char[]{'a', 'b', 'c', 'd', 'e', 'f'}),

    // SEGMENT 7--
    // Sacherschliessung
    MAB700(700, new char[]{' ', 'a', 'b', 'c',
            'd', 'e', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q',
            'r', 's',
            't', 'u', 'v', 'w', 'z'}),
    MAB705(705, new char[]{' ', 'a', 'b'},
        new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
            'l', 'm', 't', 'A'}),
    MAB710(710, new char[]{' ', 'a', 'b', 'c', 'd', 'f', 'z'}),
    MAB711(711, new char[]{' ', 'a', 'b', 'c', 'd', 'f', 'z'},
        new char[]{'a', 'v', 'x', 'y'}),
    MAB720(720, new char[]{' ', 'a', 'b', 'c', 'd'}),
    MAB730(730, new char[]{' '}),
    MAB740(740, new char[]{'p', 'c', 'f', 'n', 't', 's', 'g', 'u'}),
    MAB750(750, new char[]{' ', 'a', 'b', 'c',
            'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm'}),
    MAB751(751, new char[]{' '}),
    MAB752(752, new char[]{' '}),
    MAB753(753, new char[]{' ', 'a', 'b', 'c',
            'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm'}),
    MAB754(754, new char[]{' '}),
    MAB755(755, new char[]{' '}),
    MAB756(756, new char[]{' ', 'a', 'b', 'c',
            'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm'}),
    MAB757(757, new char[]{' '}),
    MAB758(758, new char[]{' '}),

    // SEGMENT 8--
    // Nichtstandardmaessige Nebeneintragungen
    MAB800(800, new char[]{' '}),
    MAB801(801, new char[]{' '}),
    MAB802(802, new char[]{' '}),
    MAB803(803, new char[]{' '}),
    MAB804(804, new char[]{' '}),
    MAB805(805, new char[]{' ', 'a', 'b'}),

    MAB806(806, new char[]{' '}),
    MAB807(807, new char[]{' '}),
    MAB808(808, new char[]{' '}),
    MAB809(809, new char[]{' '}),
    MAB810(810, new char[]{' '}),
    MAB811(811, new char[]{' ', 'a', 'b'}),

    MAB812(812, new char[]{' '}),
    MAB813(813, new char[]{' '}),
    MAB814(814, new char[]{' '}),
    MAB815(815, new char[]{' '}),
    MAB816(816, new char[]{' '}),
    MAB817(817, new char[]{' ', 'a', 'b'}),

    MAB818(818, new char[]{' '}),
    MAB819(819, new char[]{' '}),
    MAB820(820, new char[]{' '}),
    MAB821(821, new char[]{' '}),
    MAB822(822, new char[]{' '}),
    MAB823(823, new char[]{' ', 'a', 'b'}),

    MAB824(824, new char[]{' '}),
    MAB825(825, new char[]{' '}),
    MAB826(826, new char[]{' '}),
    MAB827(827, new char[]{' '}),
    MAB828(828, new char[]{' '}),
    MAB829(829, new char[]{' ', 'a', 'b'}),

    // SEGMENT 9--
    // RSWK-Schlagwortketten
    MAB900(900, new char[]{' '}),
    MAB902(902, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB903(903, new char[]{' '}),
    MAB904(904, new char[]{' ', 'a'}),

    MAB905(905, new char[]{' '}),
    MAB907(907, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB908(908, new char[]{' '}),
    MAB909(909, new char[]{' ', 'a'}),

    MAB910(910, new char[]{' '}),
    MAB912(912, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB913(913, new char[]{' '}),
    MAB914(914, new char[]{' ', 'a'}),

    MAB915(915, new char[]{' '}),
    MAB917(917, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB918(918, new char[]{' '}),
    MAB919(919, new char[]{' ', 'a'}),

    MAB920(920, new char[]{' '}),
    MAB922(922, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB923(923, new char[]{' '}),
    MAB924(924, new char[]{' ', 'a'}),

    MAB925(925, new char[]{' '}),
    MAB927(927, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB928(928, new char[]{' '}),
    MAB929(929, new char[]{' ', 'a'}),

    MAB930(930, new char[]{' '}),
    MAB932(932, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB933(933, new char[]{' '}),
    MAB934(934, new char[]{' ', 'a'}),

    MAB935(935, new char[]{' '}),
    MAB937(937, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB938(938, new char[]{' '}),
    MAB939(939, new char[]{' ', 'a'}),

    MAB940(940, new char[]{' '}),
    MAB942(942, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB943(943, new char[]{' '}),
    MAB944(944, new char[]{' ', 'a'}),

    MAB945(945, new char[]{' '}),
    MAB947(947, new char[]{'p', 'g', 's', 'k', 'c', 'z', 'f', 't', ' '}),
    MAB948(948, new char[]{' '}),
    MAB949(949, new char[]{' ', 'a'});
    // CHECKSTYLE:ON

    /** Die Feldnummer des repräsentierten Feldes (z.B. 001). */
    private int fieldNumber;

    /**
     * Alle möglichen Indikatoren des Feldes.
     */
    private char[] indicators;

    /**
     * Alle möglichen Unterfeldkennzeichen des Feldes oder ein leeres
     * Char-Array, so das Feld keine Unterfelder enthält.
     */
    private char[] subfieldCode;

    /** True, wenn das Feld Unterfelder haben muss und keine direkten Inhalte
     *  enthalten darf.
     */
    private boolean subfieldRequired;

    private boolean required = false;
    private boolean requiredSet = false;

    private String requiredValue = null;
    private boolean requiredValueLoaded = false;

    private Set<String> satztypen;

    private boolean satztypenLoaded = false;

    /**
     * Holds mapping for reverse lookup by field number.
     */
    private static final Map<Integer, MabFeldDefinition> LOOKUP =
        new HashMap<Integer, MabFeldDefinition>();

    /**
     * Put all definition into Map for reverse lookup.
     */
    static {
        for (MabFeldDefinition def : EnumSet.allOf(MabFeldDefinition.class)) {
            LOOKUP.put(def.getFeldnummer(), def);
        }
    }

    /**
     * Configuration containing additional information about MAB categories.
     */
    private static Configuration mabInfo;

    /**
     * Erzeugt eine MabFeldDefinition, in der alles zugelassen ist (also
     * Indikatoren und Unterfeldkennzeichen a-z, A-Z, 0-9 und Space und kein
     * Unterfeldzwang existiert).
     * @param feldnummer Die Feldnummer des Feldes.
     */
    private MabFeldDefinition(final int feldnummer) {
        this(feldnummer, MabFeld.ALL, MabFeld.ALL, false);
    }

    /**
     * Erzeugt ein Feld ohne Unterfelder, mit der angegebenen Feldnummern und
     * den angegebenen Indikatoren.
     * @param feldnummer Feldnummer des repräsentierten Feldes
     * @param indikatoren Indikatoren des repräsentierten Feldes
     */
    private MabFeldDefinition(final int feldnummer, final char[] indikatoren) {
        this(feldnummer, indikatoren, new char[]{}, false);
    }

    /**
     * Erzeugt ein Feld mit Unterfeldern, dass keine direkten Inhalte enthalten
     * darf (Unterfeldzwang).
     * @param feldnummer Feldnummer des repräsentierten Feldes.
     * @param indikatoren Indikatoren des repräsentierten Feldes.
     * @param unterfeldkennzeichen Alle zugelassenen Unterfedlkennzeichen.
     */
    private MabFeldDefinition(final int feldnummer, final char[] indikatoren,
            final char[] unterfeldkennzeichen) {
        this(feldnummer, indikatoren, unterfeldkennzeichen, true);
    }

    /**
     * Konstruktor zur Erzeugung eines Feldes, wobei Feldnummer, Indikatoren,
     * Untefeldkennzeichen und der Unterfeldzwang festgelegt werden kann.
     * @param feldnummer Die Feldnummer des repräsentierten Feldes.
     * @param indikatoren Alle möglichen Indikatoren, mit denen das Feld belegt
     *        werden darf.
     * @param unterfeldkennzeichen Leeres Char-Array falls keine Unterfelder zu
     *        gelassen sind, ansonsten alle zugelassenen Unterfeldkennzeichen.
     * @param unterfeldzwang True, falls das Feld keinen direkten Inhalte
     *        enthalten darf, sondern Unterfeldern enthalten muss.
     */
    private MabFeldDefinition(final int feldnummer, final char[] indikatoren,
            final char[] unterfeldkennzeichen, final boolean unterfeldzwang) {
        this.fieldNumber = feldnummer;
        this.indicators = indikatoren;
        this.subfieldCode = unterfeldkennzeichen;
        this.subfieldRequired = unterfeldzwang;
    }

    /**
     * Gibt das MAB-Feld zurück, dass die gegebene Feldnummer repräsentiert.
     * @param feldnummer Feldnummer des repräsentierten Feldes
     * @return {@link MabFeldDefinition}, dass genauere Informationen über das
     *         Feld mit der Feldnummer beinhaltet, oder null, falls die Nummer
     *         in Titelmab nicht vorgesehen ist.
     * @see MabFeldDefinition
     */
    public static MabFeldDefinition getFeld(final int feldnummer) {
        return LOOKUP.get(feldnummer);
    }

    /**
     * Überprüft ob eine Feldnummer in Titelmab vergeben ist.
     * @param feldnummer Feldnummer des repräsentierten Feldes
     * @return true, falls die Feldnummer in Titelmab vorgesehen ist.
     */
    public static boolean isMabField(final int feldnummer) {
        return LOOKUP.containsKey(feldnummer);
    }

    /**
     * &Uuml;berpr&uuml;ft, ob das Feld MAB gerecht ist.
     * @param feldnummer Feldnummer des Feldes.
     * @param indikator indikator des Feldes.
     * @param checkUnterfeldzwang gibt an, ob ueberprueft werden soll, ob dieses
     *        Feld Unterfelder haben muss.
     * @return true, falls die Kombination von Feldnummer und Indikator in
     *         Titelmab vorgesehen ist und (so Überprüfung erwünscht) kein
     *         Unterfeldzwang besteht.
     */
    public static boolean isMab2Field(final int feldnummer,
            final char indikator, final boolean checkUnterfeldzwang) {
        MabFeldDefinition feld = LOOKUP.get(feldnummer);
        return feld != null && !(checkUnterfeldzwang && feld.isSubfieldRequired()) && feld.hatIndikator(indikator);
    }

    /**
     * Überprüft, ob die Kombination aus Feldnummer, Indikator und
     * Unterfeldkennzeichen in Titelmab definiert und zulässig ist.
     * @param feldnummer Feldnummer des repräsentierten Feldes
     * @param indikator Indikator des repräsentierten Feldes
     * @param kennzeichen Kennzeichen des Unterfeldes
     * @return true, falls die Kombination aus Feldnummer, Indikator und
     *         Unterfeldkennzeichen in Titelmab vorgesehen ist.
     */
    public static boolean isMabSubfield(final int feldnummer,
            final char indikator, final char kennzeichen) {
        MabFeldDefinition feld = LOOKUP.get(feldnummer);
        return feld != null && (feld.hatIndikator(indikator) && feld.hatKennzeichen(kennzeichen));
    }

    /** Gibt die Feldnummer des repräsentierten Feldes zurück.
     * @return die Feldnummer des repräsentierten Feldes (z.B. 1 oder 331).
     */
    public int getFeldnummer() {
        return this.fieldNumber;
    }

    /**
     * Gibt die Feldnummer als String an (also z.B. "001" für 1, "030" für 30
     * oder "331" für 331).
     * @return einen String, der die Feldnummer mit führenden Nullen
     *         repräsentiert.
     * @see #getFeldnummerString(int)
     */
    public String getFeldnummerString() {
        return getFeldnummerString(this.fieldNumber);
    }

    /** statische Methode, die einen Integerwert der Feldnummer in einen
    entsprechenden String umwandelt, also bei Bedarf mit führenden Nullen
    auffüllt (1 -&gt; "001", 30 -&gt; "030", 331 -&gt; "331").
    @param feldnummer Integerwert der Feldnummer, die umgewandelt werden soll
    @return String, der die Feldnummer repräsentiert.
     */
    public static String getFeldnummerString(final int feldnummer) {
        return String.format("%1$03d", feldnummer);
    }

    /**
     * Gibt das Char-Array zurück, dass alle zugelassene Indikatoren enthält
     * zurück.
     * @return alle im Feld zugelassene Indikatoren als Char-Array.
     */
    public char[] getIndikatoren() {
        return this.indicators;
    }

    /**
     * Gibt alle zugelassenen Unterfeldkennzeichen zurück oder ein leeres
     * Char-Array, falls keine Unterfelder zugelassen sind.
     * @return Das Char-Array, dass die zugelassenen Unterfeldkennzeichen
     *         enthält.
     */
    public char[] getUnterfeldkennzeichen() {
        return this.subfieldCode;
    }

    /**
     * Gibt an, ob ein Indikator für das repräsentierte Feld zulässig ist.
     * @param indikator Indikator, der geprüft werden soll
     * @return True, falls der Indiaktor für das Feld zugelassen ist.
     */
    public boolean hatIndikator(final char indikator) {
        for (char indicator : this.indicators) {
            if (indicator == indikator) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gibt an, ob ein Unterfeldkennzeichen für das Feld zulässig ist.
     * @param kennzeichen Das Kennzeichen, dass geprüft werden soll.
     * @return True, falls das Unterfeldkennzeichen für das Feld zugelassen ist,
     *         sonst False.
     */
    public boolean hatKennzeichen(final char kennzeichen) {
        for (char aSubfieldCode : this.subfieldCode) {
            if (aSubfieldCode == kennzeichen) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gibt an, ob das Feld Unterfelder hat.
     * @return true, falls zulässige Unterfeldkennzeichen registriert sind.
     */
    public boolean hatUnterfelder() {
        return (this.subfieldCode.length > 0);
    }

    /**
     * Gibt an, ob das Feld alle Inhalte in Unterfeldern enthalten darf, oder
     * ob direkte Inhalte zugelassen sind.
     * @return True, falls alle Inhalte in Unterfeldern enthalten sein müssen,
     *         false, wenn auch direkte Inhalte zugelassen sind.
     *
     * TODO Review! Kann der Fall eintreten, das subfieldCode nicht > 0 ist? Wenn nicht, entfernen!
     */
    public boolean isSubfieldRequired() {
        return (this.subfieldRequired && this.subfieldCode.length > 0);
    }


    private boolean wiederholfaktorLoaded = false;

    private int wiederholfaktor = -1;

    /**
     * Gibt den Wiederholfaktor des Feldes zurück.
     *
     * @return int Wiederholfaktor
     */
    public int getWiederholfaktor() {
        Configuration info = getMabInfo();

        if (!wiederholfaktorLoaded) {
            wiederholfaktor =
                    info.getInt(String.format("mabfield[@id='%1$03d']/wiederholfaktor", this.fieldNumber), -1);

            wiederholfaktorLoaded = true;
        }

        if (wiederholfaktor == -1) {
            wiederholfaktor = info.getInt("mabfield[@id='default']/wiederholfaktor", 1);
        }

        return wiederholfaktor;
    }

    private boolean indikatorWiederholbarLoaded = false;

    private boolean indikatorwiederholbar = true;

    /**
     *
     */
    public boolean isIndikatorWiederholbar() {
        if (!indikatorWiederholbarLoaded) {
            Configuration info = getMabInfo();

            Boolean value = info.getBoolean(String.format(
                    "mabfield[@id='%1$03d']/indikator-wiederholbar",
                    this.fieldNumber), null);

            if (value == null) {
                value = info.getBoolean(
                        "mabfield[@id='default']/indikator-wiederholbar", true);
            }

            indikatorwiederholbar = value;
            indikatorWiederholbarLoaded = true;
        }

        return indikatorwiederholbar;
    }

    /**
     * Gibt den Namen des Feldes zurück.
     * @return String Name des MAB Feldes
     */
    public String getName() {
        Configuration info = getMabInfo();

        return info.getString(String.format("mabfield[@id='%1$03d']/name",
                this.fieldNumber), null);
    }

    /**
     *
     * @return
     *
     * TODO for now return false for fields with special rules (value != true|false); special rules are covered by other modules
     * TODO Liefere FALSE für Sätze, die nur für bestimmte Satztypen verlangt werden
     * TODO Verarbeite Werte wie "true, dependence "Indikator a""
     */
    public boolean isRequired() {
        boolean result = false;

        Configuration info = getMabInfo();

        // Wurde der Wert schon geladen?
        if (!requiredValueLoaded) {

            requiredValue = info.getString(String.format("mabfield[@id='%1$03d']/pflichtfeld", this.fieldNumber),
                    null);

            requiredValueLoaded = true;
        }

        if (!requiredSet) {
            if (requiredValue == null) {
                result = info.getBoolean("mabfield[@id='default']/pflichtfeld");
            }
            else {
                if ("true".equalsIgnoreCase(requiredValue)) {
                    // Feld ist immer ein Pflichtfeld
                    result = true;
                }
                else if ("ignore".equalsIgnoreCase(requiredValue)) {
                    // Felder die von anderen Feldern abhängen
                }
                else {
                    // Feld ist ein Pflichtfeld für bestimmte Sätze
                    // TODO review
                }
            }

            required = result;
            requiredSet = true;
        }
        else {
            result = required;
        }

        return result;
    }

    /**
     * Prüft, ob das Feld für einen bestimmten Satztyp ein Pflichtfeld ist.
     * @param satztyp
     * @return
     *
     * TODO prüfe auf gültige Zeichen für Satztyp
     */
    public boolean isRequiredForSatztyp(final String satztyp) {
        if (!requiredValueLoaded) {
            Configuration info = getMabInfo();

            requiredValue = info.getString(String.format("mabfield[@id='%1$03d']/pflichtfeld", this.fieldNumber),
                    null);

            requiredValueLoaded = true;
        }

        if (requiredValue != null && !"ignore".equalsIgnoreCase(requiredValue)) {
            if (!"true".equalsIgnoreCase(requiredValue)) {
                return requiredValue.contains(satztyp);
            }
            else {
                Set<String> satztypen = getSatztypen();
                return satztypen == null || satztypen.contains(satztyp);
            }
        }

        return false;
    }

    /**
     * Liefert die erlaubten Satztypen für ein Feld zurück.
     * @return
     */
    public Set<String> getSatztypen() {
        if (!satztypenLoaded) {
            Configuration info = getMabInfo();

            List<String> values = info.getList(String.format("mabfield[@id='%1$03d']/satztypen", this.fieldNumber), null);

            if (values != null && values.size() > 0) {
                satztypen = new HashSet<String>(values);
            }

            satztypenLoaded = true;
        }

        return satztypen;
    }

    /**
     * Erzeugt einen String, der das Feld repräsentiert.
     * @return einen String mit der Feldnummer, allen möglichen Indikatoren
     *         (durch | getrennt), gefolgt von einem $ Zeichen und allen
     *         möglichen Unterfeldkennzeichen (durch | getrennt), falls
     *         Unterfeldkennzeichen definiert sind.
     */
    @Override
    public String toString() {
        String feld = this.getFeldnummerString();
        if (this.indicators.length == 1) {
            feld += this.indicators[0];
        }
        else {
            feld += "(";
            for (int i = 0; i < this.indicators.length; i++) {
                if (i != 0) {
                    feld += "|";
                }
                feld += this.indicators[i];
            }
            feld += ")";
        }
        if (this.hatUnterfelder()) {
            if (this.subfieldCode.length == 1) {
                feld += "$" + this.subfieldCode[0];
            }
            else {
                feld += "(";
                for (int i = 0; i < this.subfieldCode.length; i++) {
                    if (i != 0) {
                        feld += "|";
                    }
                    feld += "$" + this.subfieldCode[i];
                }
                feld += ")";
            }
        }
        return feld;
    }

    /**
     * Returns and loads if necessary MAB information configuration.
     * @return Configuration containing information about MAB categories
     */
    private Configuration getMabInfo() {
        if (mabInfo == null) {
            URL mabInfoUrl = getClass().getClassLoader().getResource(
                    "de/kobv/mable/mab/mabinfo.xml");

            HierarchicalConfiguration config = null;

            try {
                config = new XMLConfiguration(mabInfoUrl);
                config.setExpressionEngine(new XPathExpressionEngine());
            }
            catch (ConfigurationException ce) {
                // TODO do something
            }

            mabInfo = config;
        }

        return mabInfo;
    }

    protected boolean isRequiredLoaded() {
        return requiredValueLoaded;
    }
}
