/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

/**
 * Exception, die von der Datenbankklasse geworfen wird, falls Vorgaben von MAB
 * nicht eingehalten wurden (z.B. falsche Länge der Feldnummer).
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 */
public class MabException extends Exception {

    /**
     * Enumeration, dass den Fehler beschreibt. So kann mit einer Exception
     * die Grundlage des Fehlers besser eingegrenzt werden.
     *
     * @author Pascal-Nicolas Becker <becker(at)zib.de>
     */
    public enum Description {

        /**
         * Nicht näher definierte MAB-Fehler.
         */
        UNKOWN,

        /**
         * Satzkennungslänge ist ungleich 24.
         */
        HEADERLENGTH,

        /**
         * Feldnummer, die nicht aus drei Zeichen besteht.
         */
        FIELDNUMBERLENGTH,

        /**
         * Erkennen des Strings, der die Feldnummer repräsentiert warf eine
         * NumberFormatException.
         * @see java.lang.Integer#parseInt(String)
         * @see java.lang.NumberFormatException
         */
        FIELDNUMBERFORMAT,

        /**
         * falls die Feldnummer nicht parsebar oder negativ war.
         */
        FIELDNUMBERUNPARSEABLE

    }

    /**
     *
     */
    private MabException.Description description;

    /**
     * Erzeugt eine MabException unbekannten Grunds.
     */
    public MabException() {
        this.description = MabException.Description.UNKOWN;
    }

    /**
     * Erzeugt eine MAB-Exception mit festgelegter Beschreibung.
     * @param desc Beschreibung der MAB-Exception
     */
    public MabException(final MabException.Description desc) {
        this.description = desc;
    }

    /**
     * Erzeugt eine MAB-Exception ohne Beschreibung, mit einer Fehlermeldung.
     * @param message Message for exception
     * @see java.lang.Exception#Exception(String)
     */
    public MabException(final String message) {
        super(message);
        this.description = MabException.Description.UNKOWN;
    }

    /**
     * Erzeugt eine MAB-Exception mit Fehlerbeschreibung und Fehlermeldung.
     * @param message Fehlermeldung
     * @param desc Ursache der MAB-Exception
     * @see java.lang.Exception#Exception(String)
     */
    public MabException(final String message,
            final MabException.Description desc) {
        super(message);
        this.description = desc;
    }

    /**
     * Erzeugt eine MAB-Exception mit einer Ursache.
     * @param cause Cause of exception
     * @see java.lang.Exception#Exception(Throwable)
     */
    public MabException(final Throwable cause) {
        super(cause);
        this.description = MabException.Description.UNKOWN;
    }

    /**
     * Erzeugt eine Mab-Exception mit einer Ursache und einer Beschreibung.
     * @param cause Cause of exception
     * @param desc Description of exception
     * @see java.lang.Exception#Exception(Throwable)
     */
    public MabException(final Throwable cause,
            final MabException.Description desc) {
        super(cause);
        this.description = desc;
    }

    /**
     * Creates MAB-Exception with message and cause exception.
     * @param message Message for exception
     * @param cause Cause of exception
     * @see java.lang.Exception#Exception(String, Throwable)
     */
    public MabException(final String message, final Throwable cause) {
        super(message, cause);
        this.description = MabException.Description.UNKOWN;
    }

    /**
     * Erzeugt eine Mab-Exception mit Beschreibung, Ursache und Fehlermeldung.
     * @param message Message for exception
     * @param cause Cause of exception
     * @param desc Description of exception
     * @see java.lang.Exception#Exception(String, Throwable)
     */
    public MabException(final String message, final Throwable cause,
            final MabException.Description desc) {
        super(message, cause);
        this.description = desc;
    }

    /**
     * Gibt die Fehlerbeschreibung zurück.
     * @return die Fehlerbeschreibung der MAB-Exception
     */
    public final Description getDescription() {
        return this.description;
    }
}
