/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.kernels;

import de.kobv.mable.common.MableProperties;
import de.kobv.mable.mab.parser.MabParser;
import de.kobv.mable.mab.xml.MabXmlParser;
import de.kobv.mable.reports.Report;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Timer;

import de.kobv.mable.util.XmlResourceBundleControl;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * Steuert die Aufnahme einer MAB-Datei in die Datenbank sowie die Analyse
 * dieser Daten. Arbeitet eng mit dem MabParser und den DBConnector zusammen.
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class CheckKernel extends AbstractKernel
        implements ApplicationContextAware {

    /**
     * Logger for CheckKernel class.
     */
    private static final Logger LOG = Logger.getLogger(CheckKernel.class);

    /**
     * TODO move out
     */
    private static final long MONITOR_INTERVAL = 1000;

    /**
     * TODO move out
     */
    private static final int LINE_WIDTH = 80;

    /**
     * TODO move out
     */
    private static final int MILLISECONDS = 1000;

    /**
     * Spring ApplicationContext.
     */
    private ApplicationContext context;

    private int totalDatasets;

    /**
     * TODO Assign through Spring.
     */
    private ResourceBundle messages = ResourceBundle.getBundle("de/kobv/mable/cli/mable-cli-messages",
            new XmlResourceBundleControl());

    @Override
    public void setApplicationContext(final ApplicationContext ac) {
        this.context = ac;
    }

    /**
     * Die zu analysierende Quelle
     */
    private Reader source;

    /**
     * Input stream for MAB2 data.
     */
    private InputStream stream;

    /**
     * Name of MAB2 file.
     */
    private String filename;

    private boolean fastMode;

    private boolean quietMode;

    private boolean mabxmlInput;

    /**
     * L&auml;dt den MAB2DBKernel zum entsprechenden Typ und zur entsprechenden
     * Quelle.
     */
    public CheckKernel() {
    }

    /**
     * Liest die Datei in die Datenbank ein und erhebt die Statistiken.
     *
     * @throws IOException Falls ein Fehler beim Einlesen der Quelle oder beim
     *                     Schreiben der Log-Datei auftritt.
     */
    public void start() throws IOException {
        DateFormat df = new SimpleDateFormat("yyyyMMdd-HHmmss");

        String name = getFilename();
        if (name == null) {
            // TODO remove catch earlier; throw Exception
            LOG.warn("Name der einzulesenden MAB-Datei konnte nicht gelesen werden");
            System.exit(-1);
        }

        Date timestamp = new Date();

        String logTimestamp = df.format(timestamp);

        MableProperties.setProperty("timestamp", timestamp);
        MableProperties.setProperty("timestampString", logTimestamp);
        MableProperties.setProperty("workpath", System.getProperty("user.dir"));
        MableProperties.setProperty("filename", name);

        processMAB2Band(name);
    }

    /**
     * Performs the check on the MAB2 data.
     * @param name
     */
    protected void processMAB2Band(final String name) {
        MabParser parser = getParser();

        try {
            try {

                ProgressMonitor monitor = null;

                if (!isQuietMode()) {
                    monitor = new ProgressMonitor();
                    monitor.setParser(parser);
                    monitor.setTotalDatasets(totalDatasets);
                }

                if (this.stream != null) {
                    parser.setSource(this.stream);
                }
                else if (this.source != null) {
                    parser.setSource(this.source);
                }
                else {
                    // TODO handle? should never happen! refactor!
                }
                long timestamp = System.currentTimeMillis();

                Timer timer = null;

                if (monitor != null) {
                    timer = new Timer("Monitor");
                    timer.schedule(monitor, 0, MONITOR_INTERVAL);
                }

                parser.parse();

                if (monitor != null) {
                    timer.cancel();
                    monitor.finish();
                }

                // TODO implement proper code for displaying total time
                double totalTime = System.currentTimeMillis() - timestamp;
                MableProperties.setProperty("totalTime", totalTime);
                String unit = "ms";
                if (totalTime > MILLISECONDS) {
                    unit = "s";
                    totalTime = totalTime / MILLISECONDS;
                }

                if (!quietMode) {
                    System.console().writer().println(StringUtils.rightPad(
                            messages.getString("mable.cli.totalTime") + " "
                            + totalTime + " " + unit, LINE_WIDTH - 1, ' '));
                }

                // TODO refactor: find better way to provide information to reports
                MableProperties.setProperty("datasetsProcessed", parser.getDatasetsProcessed());
                MableProperties.setProperty("datasetsIgnored", parser.getDatasetsIgnored());
            }
            catch (IOException ioe) {
                // TODO Refactor messages (use resource keys)
                LOG.warn("Beim Lesen der MAB-Datei oder beim Schreiben der Logdatei ist ein Fehler aufgetreten: " + ioe.getMessage(), ioe);
                LOG.warn("ABBRUCH.");
                // TODO signal to modules reject.warn("ABBRUCH. Details siehe Report.");
                LOG.warn("Beim Lesen der MAB-Datei oder beim Schreiben der Logdatei ist ein Fehler aufgetreten, Details siehe Report");
                System.exit(-1);
            }
            catch (ParseException pe) {
                // TODO Refactor messages (use resource keys)
                LOG.fatal("Es ist ein interner Fehler aufgetreten:" + pe.getMessage(), pe);
                LOG.warn("ABBRUCH.");
                // TODO signal to modules reject.warn("ABBRUCH. Details siehe Report.");
                LOG.fatal("Es ist ein interne Fehler aufgetreten: " + pe.getMessage(), pe);
                System.exit(-1);
            } // try-catch parser.parse()

            generateReports();
        }
        catch (Exception e) {
            LOG.error(e.getMessage(), e);
            LOG.error("Sätze verarbeitet: " + parser.getDatasetsProcessed());
            LOG.error("Letzter Satz: " + parser.getContentHandler().getLastDatasetId());
            System.exit(-1);
        }
    }

    /**
     * Generates report output.
     * TODO move out of kernel?
     */
    protected void generateReports() {
        Map<String, Report> checkers = context.getBeansOfType(Report.class);

        for (Map.Entry<String, Report> entry : checkers.entrySet()) {
            Report reportModule = entry.getValue();
            reportModule.generate();
        }
    }

    /**
     * Getter for MabParser.
     * @return MabParser
     */
    public MabParser getParser() {
        MabParser parser;

        if (mabxmlInput) {
            parser = ( MabXmlParser )context.getBean("mabxmlParser");
        }
        else if (fastMode) {
            parser = ( MabParser )context.getBean("fastParser");
        }
        else {
            parser = ( MabParser )context.getBean("mabParser");
        }

        return parser;
    }

    /**
     * Getter for name of MAB2 file that is processed.
     * @return String MAB2 file name
     */
    public String getFilename() {
        return filename;
    }

    /**
     * Setter for name of MAB2 file that is processed.
     * @param name Name of MAB2 file
     */
    public void setFilename(final String name) {
        this.filename = name;
    }

    /**
     * Getter for total number of datasets in MAB2 file.
     * @return int Number of datasets
     */
    public int getTotalDatasets() {
        return totalDatasets;
    }

    public void setTotalDatasets(final int total) {
        this.totalDatasets = total;
    }

    /**
     * Setter for reader for MAB2 data.
     * @param src Reader
     */
    public void setSource(final Reader src) {
        this.source = src;
    }

    /**
     * Getter for MAB2 data stream.
     * @return InputStream
     */
    public InputStream getStream() {
        return stream;
    }

    /**
     * Setter for stream for MAB2 data.
     * @param stream InputStream for MAB2 data
     */
    public void setStream(final InputStream stream) {
        this.stream = stream;
    }

    public boolean isFastMode() {
        return fastMode;
    }

    public void setFastMode(boolean fastMode) {
        this.fastMode = fastMode;
    }

    public boolean isQuietMode() {
        return quietMode;
    }

    public void setQuietMode(boolean quietMode) {
        this.quietMode = quietMode;
    }

    public boolean isMabxmlInput() {
        return mabxmlInput;
    }

    public void setMabxmlInput(boolean mabxmlInput) {
        this.mabxmlInput = mabxmlInput;
    }

}
