/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.cli;

import de.kobv.mable.mab.parser.MabDatasetFinder;
import gov.ornl.sns.cli.InvalidOptionException;
import java.io.*;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.cli.MissingOptionException;
import org.apache.log4j.Logger;

/**
 * Extracts a single dataset from a MAB2 file.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class ExtractCommand extends AbstractMableCommand {

    /**
     * Logger for this class.
     */
    private static final Logger LOG = Logger.getLogger(ExtractCommand.class);

    /**
     * Dataset identifier that needs to be found.
     */
    private String[] datasetIds;

    /**
     * Name of output file.
     */
    private String outputFile;

    /**
     * Name of input file containing dataset identifiers.
     */
    private String inputFile;

    /**
     * Input reader for dataset identifiers.
     */
    private Reader input;

    /**
     * Verifies parameters (options) for command.
     *
     * @throws MissingOptionException
     * @throws InvalidOptionException
     *
     * TODO use resources
     */
    @Override
    public void prepare() throws MissingOptionException, InvalidOptionException, MissingArgumentException {
        super.prepare();

        try {
            if (inputFile != null) {
                input = new FileReader(inputFile);
            }
        }
        catch (FileNotFoundException fnfe) {
            throw new InvalidOptionException("Input file not found.");
        }
    }

    /**
     * Performs search for dataset in MAB2 file.
     */
    @Override
    public void execute() {
        InputStream source;

        try {
            source = new FileInputStream(getFilename());
        }
        catch (FileNotFoundException fnfe) {
            LOG.error(getMessages().getString("error.fileNotFound"));
            return;
        }

        MabDatasetFinder finder = new MabDatasetFinder();
        finder.setSource(source);

        long totalTime = 0;

        try {
            long timestamp = System.currentTimeMillis();

            OutputStream output = getOutputStream();

            LOG.debug(datasetIds);

            // TODO change interface?
            finder.find(getIdentifiers(), output);

            if (!isQuietMode()) {
                System.out.println();
            }

            totalTime = System.currentTimeMillis() - timestamp;
        }
        catch (IOException e) {
            LOG.error("", e);
        }

        if (!isQuietMode()) {
            // TODO use resources
            System.console().writer().println(finder.getDatasetCounter() + " S\u00e4tze durchsucht ("
                    + totalTime + " ms)");
        }
    }

    /**
     * Returns output stream for datasets.
     * @return OutputStream for writing out datasets
     * @throws FileNotFoundException If output file cannot be created.
     */
    public OutputStream getOutputStream() throws FileNotFoundException {
        OutputStream out = null;
        if (outputFile == null) {
            out = System.out;
        }
        else {
            out = new FileOutputStream(outputFile);
        }
        return out;
    }

    /**
     * Returns set of identifiers for extracting.
     *
     * @return Set<String> dataset identifiers
     * @throws IOException if error reading identifiers occurs
     */
    public Set<String> getIdentifiers() throws IOException {
        Set<String> identifiers = new HashSet<String>();

        if (input != null) {
            BufferedReader in = new BufferedReader(input);

            String line;

            while ((line = in.readLine()) != null) {
                identifiers.add(line);
            }
        }
        else if (datasetIds != null) {
            identifiers.addAll(Arrays.asList(datasetIds));
        }
        else {
            // Try reading identifiers from console
            // ctrl-d can be used to finish manual input
            String line;

            while ((line = System.console().readLine()) != null) {
                identifiers.add(line);
            }
        }

        return identifiers;
    }

    /**
     * Getter for dataset identifiers that are searched.
     * @return String dataset identifier (MAB001)
     */
    public String[] getDatasetIds() {
        return datasetIds;
    }

    /**
     * Setter for dataset identifiers of MAB2 datasets being searched.
     * @param identifier MAB001 identifier for dataset
     */
    public void setDatasetIds(final String[] identifier) {
        this.datasetIds = identifier;
    }

    /**
     * Getter for name of output file.
     * @return Name of file
     */
    public String getOutputFile() {
        return outputFile;
    }

    /**
     * Setter for name of output file.
     * @param outputFile Name of file for dataset
     */
    public void setOutputFile(final String outputFile) {
        this.outputFile = outputFile;
    }

    /**
     * Getter for input file name.
     * @return String Name of input file
     */
    public String getInputFile() {
        return inputFile;
    }

    /**
     * Setter for input file name.
     * @param inputFile Name of input file
     */
    public void setInputFile(final String inputFile) {
        this.inputFile = inputFile;
    }

}
