/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.reports;

import de.kobv.mable.mab.extension.ErrorType;
import de.kobv.mable.mab.extension.IErrorStatistics;
import de.kobv.mable.util.XmlResourceBundleControl;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;

import org.apache.log4j.Logger;

/**
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class SatzfehlerReport extends AbstractReport {

    private String noErrorsMessage = "Es gab keine Satzfehler.";

    private ResourceBundle errorMessages = ResourceBundle.getBundle(
            "de/kobv/mable/messages", new XmlResourceBundleControl());

    private IErrorStatistics datasetErrorStatistics;

    private Set<String> excludeTypes = null;

    @Override
    public void generate() {
        Logger report = getReport();

        Map<ErrorType, Integer> result = getResult();

        StringWriter strOut = new StringWriter();

        PrintWriter out = new PrintWriter(strOut);

        out.println();
        out.println(getHeader());
        out.println();

        if (result.isEmpty()) {
            out.println(getNoErrorsMessage());
        }
        else {
            out.println(" Anzahl -> Fehlermeldung");
            int maxDigits = Integer.toString(findLongestValue(result)).length();

            for (Map.Entry<ErrorType, Integer> entry : result.entrySet()) {
                ErrorType errorType = entry.getKey();

                if ((excludeTypes == null) || !excludeTypes.contains(errorType.getClass().getCanonicalName())) {
                    out.println(String.format(" %1$" + maxDigits + "d -> (#%2$d) %3$s", entry.getValue(),
                            errorType.getCode(), getErrorMessage(errorType)));
                }
            }
        }

        report.info(strOut.toString());
    }

    protected String getErrorMessage(ErrorType errorType) {
        String errorMessage = errorType.getMessage();

        if (errorMessage == null) {
            String messageKey = errorType.getMessage();

            if (messageKey != null) {
                errorMessage = errorMessages.getString(messageKey);
            }
        }

        return errorMessage;
    }

    protected int findLongestValue(Map<ErrorType, Integer> result) {
        int largestValue = 0;
        for (Map.Entry<ErrorType, Integer> entry : result.entrySet()) {
            if (entry.getValue() > largestValue) {
                largestValue = entry.getValue();
            }
        }
        return largestValue;
    }

    public Map<ErrorType, Integer> getResult() {
        return getDatasetErrorStatistics().getDatasetErrors();
    }

    public String getNoErrorsMessage() {
        return noErrorsMessage;
    }

    public void setNoErrorsMessage(final String message) {
        this.noErrorsMessage = message;
    }

    public IErrorStatistics getDatasetErrorStatistics() {
        return datasetErrorStatistics;
    }

    public void setDatasetErrorStatistics(IErrorStatistics errorStatistics) {
        this.datasetErrorStatistics = errorStatistics;
    }

    /**
     * Liefert Fehlertypen (Klassen), die nicht mit angezeigt werden.
     */
    public Set<String> getExcludeTypes() {
        return excludeTypes;
    }

    /**
     * Setzt Fehlertypen (Klassen), die nicht mit angezeigt werden sollen.
     * @param excludeTypes Set mit Klassennamen
     */
    public void setExcludeTypes(final Set<String> excludeTypes) {
        this.excludeTypes = excludeTypes;
    }

}
