/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.extension.DatasetError;
import de.kobv.mable.mab.extension.ErrorType;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static org.junit.Assert.*;

/**
 * Created with IntelliJ IDEA.
 * User: jens
 * Date: 9/7/12
 * Time: 10:10 AM
 * To change this template use File | Settings | File Templates.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class WiederholfaktorenTest {

    private MabSatz satz;

    private Wiederholfaktoren module;

    private MockErrorStatistics errorStatistics;

    @Before
    public void before() {
        satz = new MabSatz("01024nM2.01200024      h");
        errorStatistics = new MockErrorStatistics();
        module = new Wiederholfaktoren();
        module.setErrorStatistics(errorStatistics);
    }

    @Test
    public void testRequiredFieldsForHSatz() {
        module.process(satz);

        assertEquals(5, errorStatistics.getErrors().size());

        // TODO 001, 030, 050, 070, 331
        List<DatasetError> errors = errorStatistics.getErrors();

        Set<ErrorType> errorTypes = new HashSet<ErrorType>();

        for (DatasetError error : errors) {
            errorTypes.add(error.getErrorType());
        }

        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("001", "h")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("030", "h")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("050", "h")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("070", "h")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("331", "h")));
    }

    @Test
    public void testRequiredFieldsForUSatz() {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");

        module.process(uSatz);

        assertEquals(5, errorStatistics.getErrors().size());

        // TODO 001, 010, 030, 050, 070
        List<DatasetError> errors = errorStatistics.getErrors();

        Set<ErrorType> errorTypes = new HashSet<ErrorType>();

        for (DatasetError error : errors) {
            errorTypes.add(error.getErrorType());
        }

        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("001", "u")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("010", "u")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("030", "u")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("050", "u")));
        assertTrue(errorTypes.contains(new RequiredCategoryErrorType("070", "u")));
    }

    @Test
    public void testNoMAB051inHSatz() throws Exception {
        addRequiredFieldForHSatz(satz);

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testOneMAB051() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld("051", ' ', "MAB051"));

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testTwoMAB051() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld("051", ' ', "MAB051-1"));
        satz.addFeld(new MabFeld("051", ' ', "MAB051-2"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new FieldCountErrorType("051"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    @Ignore(value="Implement test for MAB070 with different leaders.")
    public void testMAB70a() {
    }

    @Test
    public void testTooManyMAB030() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld("030", ' ', "MAB030-2")); // noch ein zweites Feld

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new FieldCountErrorType("030"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testMAB025zNichtWiederholbarerIndikator() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld("025", 'z', "ID1"));
        satz.addFeld(new MabFeld("025", 'z', "ID2"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new FieldCountErrorType("025", "z"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testMAB025zDifferentIndikatoren() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld("025", 'z', "ID1"));
        satz.addFeld(new MabFeld("025", 'a', "ID2"));
        satz.addFeld(new MabFeld("025", 'b', "ID2"));

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testWiederholfaktorOhneBegrenzung() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(655, ' ', "MAB655"));

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testHSatzMitMAB089() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(89, ' ', "MAB089"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("h", "089"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testHSatzMitMAB090() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(90, ' ', "MAB090"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("h", "090"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testHSatzMitMAB010() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(10, ' ', "MAB010"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("h", "010"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testUSatzMitMAB300() throws Exception {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");
        addRequiredFieldsForUSatz(uSatz);
        uSatz.addFeld(new MabFeld(300, ' ', "MAB300"));

        module.process(uSatz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("u", "300"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testUSatzMitMAB310() throws Exception {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");
        addRequiredFieldsForUSatz(uSatz);
        uSatz.addFeld(new MabFeld(310, ' ', "MAB310"));

        module.process(uSatz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("u", "310"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testUSatzMitMAB333() throws Exception {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");
        addRequiredFieldsForUSatz(uSatz);
        uSatz.addFeld(new MabFeld(333, ' ', "MAB333"));

        module.process(uSatz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("u", "333"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testUSatzMitMAB340() throws Exception {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");
        addRequiredFieldsForUSatz(uSatz);
        uSatz.addFeld(new MabFeld(340, ' ', "MAB340"));

        module.process(uSatz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("u", "340"), errorStatistics.getErrors().get(0).getErrorType());
    }

    @Test
    public void testUSatzMitMAB522() throws Exception {
        MabSatz uSatz = new MabSatz("01024nM2.01200024      u");
        addRequiredFieldsForUSatz(uSatz);
        uSatz.addFeld(new MabFeld(522, ' ', "MAB522"));

        module.process(uSatz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertEquals(new BadExtraFieldErrorType("u", "522"), errorStatistics.getErrors().get(0).getErrorType());
    }

    /**
     *
     * @throws Exception
     *
     * TODO Ist das, das gewünscht Verhalten oder sollte nur eine Fehlermeldung produziert werden.
     */
    @Test
    public void testHSatzMitZweiMAB010() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(10, ' ', "MAB010"));
        satz.addFeld(new MabFeld(10, ' ', "MAB010"));

        module.process(satz);

        assertEquals(2, errorStatistics.getErrors().size());
        assertEquals(new FieldCountErrorType("010"), errorStatistics.getErrors().get(0).getErrorType());
        assertEquals(new BadExtraFieldErrorType("h", "010"), errorStatistics.getErrors().get(1).getErrorType());
    }

    @Test
    public void testEinmalMAB058() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(58, ' ', "MAB058"));

        assertEquals(0, errorStatistics.getErrors().size());
    }

    /**
     *
     * @throws Exception
     *
     * TODO MABLE-81 klären
     */
    @Test
    @Ignore(value="Spezifikation sagt 10 mal erlaubt")
    public void testZweimalMAB058() throws Exception {
        addRequiredFieldForHSatz(satz);
        satz.addFeld(new MabFeld(58, ' ', "MAB058-1"));
        satz.addFeld(new MabFeld(58, ' ', "MAB058-2"));

        assertEquals(1, errorStatistics.getErrors().size());
    }

    protected void addRequiredFieldForHSatz(MabSatz satz) throws Exception {
        satz.addFeld(new MabFeld("001", ' ', "ID0001"));
        satz.addFeld(new MabFeld("030", ' ', "MAB030"));
        satz.addFeld(new MabFeld("050", ' ', "MAB050"));
        satz.addFeld(new MabFeld("070", ' ', "MAB070"));
        satz.addFeld(new MabFeld("331", ' ', "MAB331"));
    }

    protected void addRequiredFieldsForUSatz(MabSatz satz) throws Exception {
        satz.addFeld(new MabFeld("001", ' ', "ID0001"));
        satz.addFeld(new MabFeld("010", ' ', "MAB010"));
        satz.addFeld(new MabFeld("030", ' ', "MAB030"));
        satz.addFeld(new MabFeld("050", ' ', "MAB030"));
        satz.addFeld(new MabFeld("070", ' ', "MAB070"));
    }

}
