/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.validator.routines.ISBNValidator;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.*;

/**
 * Unit tests for class ValidateIsbn.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class ValidateIsbnTest {

    private String[] validValues = new String[]{
        "ISBN 3-406-30964-X",
        "ISBN 0-582-10271-5",
        "ISBN 0-582-10270-7",
        "ISBN 3-499-55546-8",
        "ISBN 0-19-814828-3",
        "ISBN 3-11-014627-4",
        "ISBN 3-89012-454-2",
        "ISBN 3-929318-33-4",
        "ISBN 3-496-02568-9",
        "ISBN 90-272-3028-5",
        "ISBN 1-55619-377-7",
        "ISBN 3-518-37825-2",
        "ISBN 0-8039-8956-3",
        "ISBN 0-521-47880-4",
        "ISBN 3-437-11667-3",
//        "ISBN 978-3-9811109-4-4 kart.",
        "ISBN 0-87645-080-X",
        "ISBN 3-8062-1266-X",
        "ISBN 3-921388-35-X",
        "ISBN 88-86712-09-X",
        "ISBN 3-499-16230-X",
        "ISBN 3-85435-116-X",
        "ISBN 3-8061-0610-X",
        "ISBN 81-208-1018-X",
        "ISBN 975-585-505-X"
//        "ISBN 978-981-081402-1", // TODO legal or not
//        "ISBN 981-081402-X" // TODO legal or not
    };

    private String[] invalidValues = new String[] {
        "ISBN 13 978-0-19-516839-6",
        "ISBN 13-978-0-268-03490-0",
        "ISBN 13-978-0-268-03491-7",
        "ISBN 13-978-0-321-35668-0",
        "ISBN 13-978-0-470-05902-9"
    };

    private ValidateIsbn module;

    private MockErrorStatistics errorStatistics;

    private MockIsbnStatistics isbnStatistics;

    private MabSatz satz;

    @Before
    public void before() {
        satz = new MabSatz("01024nM2.01200024      h");
        errorStatistics = new MockErrorStatistics();
        isbnStatistics = new MockIsbnStatistics();
        module = new ValidateIsbn();
        module.setErrorStatistics(errorStatistics);
        module.setIsbnStatistics(isbnStatistics);
    }

    @Test
    public void testIsValid() {
        ValidateIsbn module = new ValidateIsbn();

        for (String isbn : validValues) {
            assertTrue("Value '" + isbn + "' should be valid.",
                    module.isValid(isbn));
        }

        for (String isbn : invalidValues) {
            assertFalse("Value'" + isbn + "' should not be valid.",
                    module.isValid(isbn));
        }
    }

    @Test
    public void testPrepareValue() {
        ValidateIsbn module = new ValidateIsbn();

        assertEquals("3-437-11667-3", module.prepareValue("ISBN 3-437-11667-3"));
        assertEquals("3-437-11667-3", module.prepareValue("ISBN 3-437-11667-3 Pp."));
        assertEquals("3-437-11667-3", module.prepareValue("ISBN 3-437-11667-3 (alk. paper)"));
        assertEquals("3-437-11667-3", module.prepareValue("ISBN 3-437-11667-3 Pp. : DM 29.80"));
    }

    @Test
    // @Ignore
    public void testValidateISBN() {
        ISBNValidator validator = ISBNValidator.getInstance();

//        String value = "978-981-081402-1";
//        String value = "981-081402-X";
        String value = "981-081402-X";
        assertTrue(validator.isValid(StringUtils.remove(value, "-")));
    }

    @Test
    public void testValidIsbnIn540a() {
        satz.addFeld(new MabFeld(540, 'a', "ISBN 3-929318-33-4"));

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testInvalidIsbnIn540a() {
        satz.addFeld(new MabFeld(540, 'a', "ISBN 13 978-0-19-516839-6"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
    }

    @Test
    public void testValidIsbnIn540b() {
        satz.addFeld(new MabFeld(540, 'b', "ISBN 3-929318-33-4"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
    }

    @Test
    public void testInvalidIsbnIn540b() {
        satz.addFeld(new MabFeld(540, 'b', "ISBN 13 978-0-19-516839-6"));

        module.process(satz);

        assertEquals(0, errorStatistics.getErrors().size());
    }


    @Test
    public void testIsbnWithoutPrefix() {
        satz.addFeld(new MabFeld(540, 'a', "3-929318-33-4"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
        assertTrue(errorStatistics.getErrors().get(0) instanceof IsbnError);

        IsbnError error1 = ( IsbnError )errorStatistics.getErrors().get(0);

        assertEquals(IsbnErrorType.INVALID_ISBN_IN_540A, (( IsbnErrorType )error1.getErrorType() ).getMessageKey());
        // TODO assertEquals(IsbnErrorType.ISBN_WITHOUT_LEADER, (( IsbnErrorType )error2.getErrorType() ).getMessageKey());
    }

    @Test
    public void testIsbnIn540z() {
        satz.addFeld(new MabFeld(540, 'z', "ISBN"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());
    }

    @Test
    public void testIsbnWithSmallX() {
        satz.addFeld(new MabFeld(540, 'a', "ISBN 88-86712-09-x"));

        module.process(satz);

        assertEquals(1, errorStatistics.getErrors().size());

        assertTrue(errorStatistics.getErrors().get(0) instanceof IsbnError);

        IsbnError error1 = ( IsbnError )errorStatistics.getErrors().get(0);

        assertEquals(IsbnErrorType.INVALID_ISBN_IN_540A, (( IsbnErrorType )error1.getErrorType() ).getMessageKey());
        // TODO assertEquals(IsbnErrorType.ISBN_SMALL_X, (( IsbnErrorType )error2.getErrorType() ).getMessageKey());
    }

}
