/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.extension.DefaultReferencesStore;
import de.kobv.mable.mab.extension.ErrorType;
import de.kobv.mable.mab.extension.ReferencesStore;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import static org.junit.Assert.assertEquals;

/**
 * Unit tests for class Reference.
 *
 * Das Reference Modul ist darauf angewiesen, daß die Klasse ReferencesHolder die IDs von allen Sätzen einsammelt.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO ways to isolate tests from other classes (ReferencesHolder, ReferencesStore)?
 */
public class ReferenceTest {

    private Reference module;

    private ReferencesHolder holder;

    private MockErrorStatistics errorStatistics;

    private MabSatz uSatz;

    private MabSatz hSatz;

    private ReferencesStore referenceStore;

    @Before
    public void before() {
        uSatz = new MabSatz("01024nM2.01200024      u");
        uSatz.addFeld(new MabFeld(1, ' ', "ID00u"));
        uSatz.addFeld(new MabFeld(10, ' ', "ID00h"));

        hSatz = new MabSatz("01024nM2.01200024      h");
        hSatz.addFeld(new MabFeld(1, ' ', "ID00h"));

        errorStatistics = new MockErrorStatistics();
        referenceStore = new DefaultReferencesStore();

        holder = new ReferencesHolder();
        holder.setReferencesStore(referenceStore);

        module = new Reference();
        module.setErrorStatistics(errorStatistics);
        module.setReferencesStore(referenceStore);
        module.setSatzTyp('u');
        module.setFeldnummer(10);
        module.setTargetDatasetType("h");
    }

    @Test
    public void testDefaultErrorMessage() {
        module.process(uSatz);
        module.finish();

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals("In Feld 010 referenzierter Satz wurde nicht gefunden.", errorType.getMessage());
    }

    @Test
    public void testCustomErrorMessage() {
        module.setErrorMessageKey("FEHLERHAFTE_VERKNUEPFUNG");
        module.process(uSatz);
        module.finish();

        assertEquals(1, errorStatistics.getErrors().size());

        ErrorType errorType = errorStatistics.getErrors().get(0).getErrorType();

        assertEquals("Es fehlen referenzierte Daten (z.B. ein H-Satz, der im MAB-Feld 010 eines U-Satzes referenziert" +
                " wurde).", errorType.getMessage());
    }

    @Test
    public void testReferenceValid() {
        holder.process(hSatz);
        module.process(hSatz);
        holder.process(uSatz);
        module.process(uSatz);
        module.finish();

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testReferenceValidReverseOrderOfDatasets() {
        holder.process(uSatz);
        module.process(uSatz);
        holder.process(hSatz);
        module.process(hSatz);
        module.finish();

        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testReferenceBroken() {
        holder.process(uSatz);
        module.process(uSatz);
        module.finish();

        assertEquals(1, errorStatistics.getErrors().size());
    }


}
