/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.MabSatzkennung;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Test module for checking length of a field.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO Create framework that makes asserting specific errors easy.
 */
public class FieldLengthTest {
    
    private MabSatz satz;

    private FieldLength module;

    private MockErrorStatistics errorStatistics;
    
    @Before
    public void before() {
        satz = new MabSatz(new MabSatzkennung(0, "h", "n"));
        errorStatistics = new MockErrorStatistics();
        module = new FieldLength();
        module.setErrorStatistics(errorStatistics);
        module.setFeldnummer(51);
        module.setMax(8);
        module.setMin(7);
    }

    /**
     * When checking a field of correct length no error should be generated.
     */
    @Test
    public void testFieldCorrectLength() {
        satz.addFeld(new MabFeld(51, ' ', "1234567"));
        module.process(satz);
        assertEquals(0, errorStatistics.getErrors().size());
    }

    /**
     * An error should be generated when field content is too long.
     */
    @Test
    public void testFieldTooLong() {
        satz.addFeld(new MabFeld(51, ' ', "123456789"));
        module.process(satz);
        assertEquals(1, errorStatistics.getErrors().size());
    }

    /**
     * An error should be generated when field content is too short.
     */
    @Test
    public void testFieldTooShort() {
        satz.addFeld(new MabFeld(51, ' ', "123456"));
        module.process(satz);
        assertEquals(1, errorStatistics.getErrors().size());
    }

    /**
     * Test setting a specific field length and a field with correct length.
     */
    @Test
    public void testFieldWithSpecificLength() {
        module.setLength(15);
        satz.addFeld(new MabFeld(51, ' ', "0123456789ABCDE"));
        module.process(satz);
        assertEquals(0, errorStatistics.getErrors().size());
    }

    /**
     * Test setting a specific field length and checking a field that is too
     * short.
     */
    @Test
    public void testFieldWithSpecificLengthTooShort() {
        module.setLength(15);
        satz.addFeld(new MabFeld(51, ' ', "01234"));
        module.process(satz);
        assertEquals(1, errorStatistics.getErrors().size());
    }

    @Test
    public void testFieldEmpty() {
        satz.addFeld(new MabFeld(51, ' '));
        module.process(satz);
        assertEquals(1, errorStatistics.getErrors().size());
    }

    @Test
    public void testFieldEmptyWithMinZero() {
        satz.addFeld(new MabFeld(51, ' '));
        module.setMin(0);
        module.process(satz);
        assertEquals(0, errorStatistics.getErrors().size());
    }

    @Test
    public void testGetLength() {
        module.setLength(10);
        assertEquals(10, module.getLength());
    }

    @Test
    public void testGetMin() {
        module.setMin(4);
        assertEquals(4, module.getMin());
    }

    @Test
    public void testGetMax() {
        module.setMax(9);
        assertEquals(9, module.getMax());
    }

}
