/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.IMabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.extension.DatasetError;
import de.kobv.mable.mab.extension.ErrorType;
import de.kobv.mable.mab.extension.ReferencesStore;
import de.kobv.mable.mab.extension.ReferencesStoreListener;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;

/**
 * Verifies existence of reference.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO Cannot handle targetDatasetType = null (any type)
 */
public class Reference extends AbstractMabCheck implements ReferencesStoreListener {

    /**
     * Stores missing datasets and what datasets referenced them.
     *
     * If two datasets references the same dataset both errors are stored
     * separately.
     */
    private Map<String, Set<String>> missingSatzIds = new HashMap<String, Set<String>>();

    /**
     *
     */
    private String targetDatasetType = null;

    /**
     * Store for dataset references.
     */
    private ReferencesStore referencesStore;

    public Reference() {
    }

    @Override
    public void process(final MabSatz satz) {
        if (getSatzTyp() == 0 || getSatzTyp() == satz.getSatzkennung().getTyp()) {

            List<IMabFeld> felder = satz.getFelder(getKategorie(), getIndikator());

            for (IMabFeld feld : felder) {
                checkField(satz.getId(), feld);
            }
        }
    }

    protected void checkField(String satzId, final IMabFeld feld) {
        // Prüfung, ob Reference ID bereits gefunden wurde
        String referenceId = feld.getInhalt();

        if (!referencesStore.isDatasetExists(referenceId, targetDatasetType)) {
            // Nicht gefunden, daher zur Suchliste hinzufügen
            Set<String> saetze = missingSatzIds.get(referenceId);
            if (saetze == null) {
                saetze = new HashSet<String>();
                missingSatzIds.put(referenceId, saetze);
            }
            saetze.add(satzId);
        }
    }

    /**
     * Log missing references somewhere?
     */
    @Override
    public void finish() {
        for (Map.Entry<String, Set<String>> entry : missingSatzIds.entrySet()) {
            for (String datasetId : entry.getValue()) {
                getErrorStatistics().addError(new DatasetError(getErrorType(), datasetId));
            }
        }
    }

    /**
     * Getter for bean that stores dataset references.
     * @return ReferencesStore
     */
    public ReferencesStore getReferencesStore() {
        return referencesStore;
    }

    /**
     * Setter for bean that stores dataset references.
     * @param store ReferencesStore
     */
    @Autowired(required=true)
    public void setReferencesStore(final ReferencesStore store) {
        this.referencesStore = store;

        // TODO use weak listener to make sure garbage collection works
        this.referencesStore.addListener(this);
    }

    /**
     *
     * @return
     *
     * TODO solution?
     */
    public String getErrorMessage() {
        // String message = super.getErrorMessage();
        return null; // String.format(message, getFieldNumber());
    }

    @Override
    public void datasetFound(final String datasetId, final String type) {
        if ((type == null) || type.equals(targetDatasetType)) {
            missingSatzIds.remove(datasetId);
        }
    }

    public String getTargetDatasetType() {
        return targetDatasetType;
    }

    public void setTargetDatasetType(final String targetType) {
        this.targetDatasetType = targetType;
    }

    public ErrorType getErrorType() {
        ErrorType errorType = super.getErrorType();
        if (errorType == null) {
            errorType =  new ReferenceErrorType(getKategorie(), getTargetDatasetType(), getErrorMessageKey(),
                    getErrorCode());
            setErrorType(errorType);
        }
        return errorType;
    }
}
