/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.extension.ErrorType;
import org.apache.commons.lang.builder.HashCodeBuilder;

import java.util.Set;

/**
 * Created with IntelliJ IDEA.
 * User: jens
 * Date: 8/23/12
 * Time: 5:12 PM
 * To change this template use File | Settings | File Templates.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO use int or enum for ISBN error type
 * TODO review/refactor Verwendung von String Konstanten (Enum, Unterklassen?)
 * TODO Review Design (Siehe unten)
 * Sollten die Informationen über mögliche Fehlerursachen bei den ISBN Einträgen Teil der Fehlerklasse (IsbnError) oder
 * sollten sie Teil der Klasse IsbnErrorType sein. Es soll kein Fehlertyp für jede mögliche Kombination von Ursachen
 * erzeugt werden. Der Fehler ist "eine inkorrekter ISBN Eintrag". Die möglichen Ursachen beziehen sich auf den
 * konkreten Eintrag, allerdings lassen auch sie sich in "Typen" einteilen, wobei ein falscher Eintrag mehrere Ursachen
 * haben kann.
 *
 * TODO Review Reason... und entferne gegebenenfalls.
 */
public class IsbnErrorType extends ErrorType {

    /**
     * Formal inkorrekte ISBN in Feld 540a.
     */
    public final static String INVALID_ISBN_IN_540A = "INVALID_ISBN_IN_540A";

    /**
     * Formal korrekte ISBN in Feld 540b.
     */
    public final static String VALID_ISBN_IN_540B = "VALID_ISBN_IN_540B";

    /**
     * ISBN Eintrag in Feld 540z.
     */
    public final static String ISBN_IN_540Z = "ISBN_IN_540Z";

    /**
     * Enum für Ursachen für inkorrekte ISBN Einträge.
     */
    public enum Reason {

        /**
         * Präfix "ISBN " fehlt.
         */
        ISBN_WITHOUT_LEADER("ISBN_WITHOUT_LEADER"),

        /**
         * Formatierung entspricht nicht der Spezifikation.
         */
        ISBN_BAD_FORMAT("ISBN_BAD_FORMAT"),

        /**
         * Es wurde ein kleines 'x' gefunden.
         */
        ISBN_SMALL_X("ISBN_BAD_FORMAT_SMALL_X");

        /**
         * Schlüssel für Fehlermeldung.
         */
        private String messageKey;

        /**
         * Constructor für Reason.
         * @param messageKey Schlüssel für Fehlermeldung
         */
        Reason(String messageKey) {
            this.messageKey = messageKey;
        }

        /**
         * Liefert Schlüssel für Fehlermeldung zurück.
         * @return String Schlüssel
         */
        public String getMessageKey() {
            return messageKey;
        }

        public String getMessage() {
            return ErrorType.getMessage(getMessageKey());
        }

    }

    /**
     * Schlüssel für Fehlermeldung.
     */
    private String messageKey;

    /**
     * Gefundene Gründe für inkorrekten ISBN Eintrag.
     */
    private Set<Reason> reasons;

    /**
     * Constructs IsbnErrorType für Schlüssel.
     * @param messageKey
     */
    public IsbnErrorType(final String messageKey) {
        super(messageKey);
        this.messageKey = messageKey;
    }

    public IsbnErrorType(final String messageKey, final Set<Reason> reasons) {
        super(messageKey);
        this.messageKey = messageKey;
        this.reasons = reasons;
    }

    /**
     * Liefert den Schlüssel für die Fehlermeldung.
     * @return String Ressourcen-Schlüssel für Fehlermeldung
     */
    public String getMessageKey() {
        return messageKey;
    }

    /**
     * Liefert Fehlermeldung
     * @return String Fehlermeldung
     */
    @Override
    public String getMessage() {
        return super.getMessage();
    }

    /**
     * Liefert mögliche Gründe für Fehler.
     * @return
     */
    public Set<Reason> getReasons() {
        return reasons;
    }

    /**
     * Berechnet Hash für Fehlertyp.
     *
     * Der Hash basiert auf der Klasse und dem Schlüssel für die Fehlermeldung. Die Gründe für die Fehlermeldung werden
     * nicht mit einbezogen.
     *
     * @return int
     */
    @Override
    public int getHash() {
        return new HashCodeBuilder()
                .append(getClass())
                .append(messageKey)
                .toHashCode();
    }

}
