/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2.modules;

import de.kobv.mable.mab.IMabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.extension.DatasetError;

import java.util.List;

/**
 * MableModule the length of a field.
 *
 * The module can be configured with a minimum and maximum length for a field
 * or an exact length that needs to be matched. Setting the required length is
 * setting the minimum and the maximum property to the same value.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class FieldLength extends AbstractMabCheck {

    /**
     * Minimum length of field.
     */
    private int min;

    /**
     * Maximum length of field.
     */
    private int max;

    /**
     * Exact required length of field.
     */
    private int length;

    /**
     * Checks the length of the field.
     *
     * TODO checking the field number should not be necessary (assert)
     * TODO add method to parent class for adding error
     */
    @Override
    public void process(final MabSatz satz) {
        List<IMabFeld> felder = satz.getFelder(getKategorie(), getIndikator());

        for (IMabFeld feld : felder) {
            String content = feld.getInhalt();

            int contentLength = 0;

            if (content != null) {
                contentLength = content.length();
            }

            if (contentLength < min || contentLength > max) {
                getErrorStatistics().addError(new DatasetError(new FieldLengthErrorType(getKategorie()), satz.getId()));
            }
        }
    }

    /**
     * Getter for required length for field.
     * @return Configured required length for field
     */
    public int getLength() {
        return length;
    }

    /**
     * Setter for required length for field.
     * @param len Required length for field
     */
    public void setLength(final int len) {
        this.length = len;
        setMin(length);
        setMax(length);
    }

    /**
     * Getter for maximum length of field.
     * @return Configured maximum length for field
     */
    public int getMax() {
        return max;
    }

    /**
     * Setter for maximum length of field.
     * @param maxLength Maximum length for field (int)
     */
    public void setMax(final int maxLength) {
        this.max = maxLength;
    }

    /**
     * Getter for minimum length of field.
     * @return Configured minimum length as int
     */
    public int getMin() {
        return min;
    }

    /**
     * Setter for minimum length of field.
     * @param minLength Minimum length for field (int)
     */
    public void setMin(final int minLength) {
        this.min = minLength;
    }

}
