/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab2;

import de.kobv.mable.common.MableConfigurationException;
import de.kobv.mable.config.MableConfig;
import de.kobv.mable.mab.MabFeld;
import de.kobv.mable.mab.MabSatz;
import de.kobv.mable.mab.MabSatzkennung;
import de.kobv.mable.mab.MabUnterfeld;
import de.kobv.mable.mab.extension.ErrorStatistics;
import de.kobv.mable.mab.extension.MableModule2;
import de.kobv.mable.mab.parser.ContentHandler;
import de.kobv.mable.mab2.modules.DatasetStatistics;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import java.util.Set;

/**
 * New implementation for handling entire MAB datasets.
 */
public class DefaultMabDataHandler implements ContentHandler {

    /**
     * Logger for this class.
     */
    private static final Logger LOG = Logger.getLogger(DefaultMabDataHandler.class);

    /**
     * Dataset currently being read.
     */
    private MabSatz currentDataset;

    private MabFeld currentField;

    private MableConfig config;

    private String currentDatasetId;

    private String lastDatasetId;

    private Set<MableModule2> modules;

    private ErrorStatistics errorStatistics;

    private MableModule2 datasetStatistics;

    @Override
    public void startParsing() throws MableConfigurationException {
        modules = config.getModules();
        for (MableModule2 module : modules) {
            module.init();
            if (module instanceof DatasetStatistics) {
                datasetStatistics = module;
            }
        }
    }

    @Override
    public void endParsing() {
        for (MableModule2 module : modules) {
            module.finish();
        }
    }

    @Override
    public void startSatz(MabSatzkennung satzkennung) {
        currentDataset = new MabSatz(satzkennung.getValue());
    }

    @Override
    public void endSatz() {
        // TODO get bytes for current dataset?
        processDataset(currentDataset);
        lastDatasetId = currentDatasetId;
        currentDatasetId = null;
        errorStatistics.setCurrentDataset001(currentDatasetId);
    }

    @Override
    public void startFeld(String fieldNumber, char indicator, String content) {
        assert(fieldNumber.length() == 3);

        int iFeldnummer;

        try {
            iFeldnummer = Integer.parseInt(fieldNumber);
        }
        catch (NumberFormatException nfe) {
            iFeldnummer = -1;
        }

        if (iFeldnummer >= 1) {
            // TODO refactor? too much responsibility?
            if (iFeldnummer == 1) {
                currentDatasetId = content;
                errorStatistics.setCurrentDataset001(currentDatasetId);
            }

            currentField = new MabFeld(iFeldnummer, indicator, content);

            currentDataset.addFeld(currentField);
        }
        else {
            // TODO what if Feldnummer is not numeric?
        }
    }

    @Override
    public void startUnterfeld(char kennzeichen, String content) {
        currentField.addSubfield(new MabUnterfeld(kennzeichen, content));
    }

    @Override
    public MabSatz getCurrentDataset() {
        return currentDataset;
    }

    public void setConfig(MableConfig config) {
        this.config = config;
    }

    /**
     * Process MAB dataset.
     *
     * @param satz
     */
    protected void processDataset(MabSatz satz) {
        boolean deleted = satz.getSatzkennung().getStatus() == 'd';

        if (deleted && datasetStatistics != null) {
            datasetStatistics.process(satz);
        }
        else {
            for (MableModule2 module : modules) {
                module.process(satz);
            }
        }
    }

    @Autowired
    @Required
    public void setErrorStatistics(ErrorStatistics errorStatistics) {
        this.errorStatistics = errorStatistics;
    }

    @Override
    public String getLastDatasetId() {
        return lastDatasetId;
    }

}
