/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import static org.junit.Assert.*;
import org.junit.Test;

/**
 * Unit tests for class MabSatzkennung.
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class MabSatzkennungTest {

    /**
     * Valid MAB dataset header for testing.
     */
    public static final String VALID_KENNUNG = "01024nM2.01200024      h";

    /**
     * Invalid MAB dataset header for testing.
     */
    public static final String INVALID_KENNUNG = "00abcaM2.12300020   3  a";

    @Test
    public void testGetValue() {
        MabSatzkennung kennung = new MabSatzkennung(VALID_KENNUNG);
        assertEquals(VALID_KENNUNG, kennung.getValue());
    }

    @Test
    public void testIsLengthValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isLengthValid());
    }

    @Test
    public void testIsLengthValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isLengthValid());
    }

    @Test
    public void testGetStatus() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertEquals('n', kennung.getStatus());
    }

    @Test
    public void testIsStatusValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isStatusValid());
    }

    @Test
    public void testIsStatusValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isStatusValid());
    }

    @Test
    public void testIsVersionValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isVersionValid());
    }

    @Test
    public void testIsVersionValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isVersionValid());
    }

    @Test
    public void testIsIndicatorLengthValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isIndicatorLengthValid());
    }

    @Test
    public void testIsIndicatorLengthValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isIndicatorLengthValid());
    }

    @Test
    public void testIsTeilfeldkennungslaengeValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isTeilfeldkennungslaengeValid());
    }

    @Test
    public void testIsTeilfeldkennungslaengeValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isTeilfeldkennungslaengeValid());
    }

    @Test
    public void testIsDataStartAddressValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isDataStartAddressValid());
    }

    @Test
    public void testIsDataStartAddressValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isDataStartAddressValid());
    }

    @Test
    public void testIsReservedFieldsEmpty() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isReservedFieldsEmpty());
    }

    @Test
    public void testIsReservedFieldsEmptyWithNonEmptyFields() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isReservedFieldsEmpty());
    }

    @Test
    public void testGetType() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertEquals('h', kennung.getTyp());
    }

    @Test
    public void testIsTypeValid() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(VALID_KENNUNG);
        assertTrue(kennung.isTypeValid());
    }

    @Test
    public void testIsTypeValidWithInvalidResult() {
        MabSatzkennung kennung = new MabSatzkennung();
        kennung.setValue(INVALID_KENNUNG);
        assertFalse(kennung.isTypeValid());
    }

    @Test
    public void testConstructSatzkennung() {
        String kennung = MabSatzkennung.constructSatzkennung(256, "h", "n");

        assertEquals("00256nM2.01200024      h", kennung);
    }

    @Test
    public void testIsValidTrue() {
        MabSatzkennung kennung = new MabSatzkennung(VALID_KENNUNG);
        assertTrue(kennung.isValid());
    }

    @Test
    public void testIsValidFalse() {
        MabSatzkennung kennung = new MabSatzkennung(INVALID_KENNUNG);
        assertFalse(kennung.isValid());
    }

}
