/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO verify and extend getFelder tests
 */
public class MabSatzTest {

    /**
     * Valid MAB dataset header for testing.
     */
    public static final String VALID_KENNUNG = "01024nM2.01200024      h";

    private MabSatz satz;

    @Before
    public void before() {
        satz = new MabSatz(VALID_KENNUNG);
    }

    @Test
    public void testConstructMabSatz() {
        assertEquals(VALID_KENNUNG, satz.getKennung());
    }

    @Test
    public void testConstructMabSatzWithFields() {
        MabFeld feld = new MabFeld(331, 'i');
        List<IMabFeld> felder = new ArrayList<IMabFeld>();
        felder.add(feld);
        MabSatz satz = new MabSatz(VALID_KENNUNG, felder);
        assertEquals(VALID_KENNUNG, satz.getKennung());
        assertNotNull(satz.getFelder());
        assertEquals(1, satz.getFelder().size());
        assertTrue(satz.getFelder().contains(feld));
    }

    @Test
    public void testAddFeld() {
        MabFeld feld = new MabFeld(331, 'i');
        satz.addFeld(feld);

        assertNotNull(satz.getFelder());
        assertEquals(1, satz.getFelder().size());
        assertTrue(satz.getFelder().contains(feld));
    }

    @Test
    public void testAddSecondFeld() {
        List<IMabFeld> felder = new ArrayList<IMabFeld>();

        MabFeld feld = new MabFeld(331, 'i');
        felder.add(feld);

        MabSatz satz = new MabSatz(VALID_KENNUNG, felder);

        assertNotNull(satz.getFelder());
        assertEquals(1, satz.getFelder().size());
        assertTrue(satz.getFelder().contains(feld));

        MabFeld feld2 = new MabFeld(25, 'z');
        satz.addFeld(feld2);
        assertEquals(2, satz.getFelder().size());

        felder = satz.getFelder();
        assertTrue(felder.contains(feld));
        assertTrue(felder.contains(feld2));
    }

    @Test
    public void testGetFelder() {
        MabFeld feld = new MabFeld(331, 'i');
        List<IMabFeld> felder = new ArrayList<IMabFeld>();
        felder.add(feld);
        MabSatz satz = new MabSatz(VALID_KENNUNG, felder);
        assertNotNull(satz.getFelder());
        assertEquals(1, satz.getFelder().size());
        assertTrue(satz.getFelder().contains(feld));
    }

    @Test
    public void testGetKennung() {
        assertEquals(VALID_KENNUNG, satz.getKennung());
    }

    @Test
    public void testToString() {
        MabSatz satz = new MabSatz(VALID_KENNUNG);
        String output = satz.toString();
        assertNotNull(output);
    }

    @Test
    public void testToStringWithFields() {
        MabFeld feld = new MabFeld(331, 'i');
        List<IMabFeld> felder = new ArrayList<IMabFeld>();
        felder.add(feld);
        MabSatz satz = new MabSatz(VALID_KENNUNG, felder);
        String output = satz.toString();
        assertNotNull(output);
    }

    @Test
    public void testRemoveFeld() {
        satz.addFeld(new MabFeld(331, 'i'));

        List<IMabFeld> felder = satz.getFelder();

        assertEquals(1, felder.size());

        satz.removeFeld(felder.get(0));

        assertEquals(0, satz.getFelder().size());
    }

    @Test
    public void testRemoveFeldNotPresent() {
        satz.removeFeld(new MabFeld(331, 'i'));
    }

    @Test
    public void testGetSatzkennung() {
        MabSatzkennung kennung = satz.getSatzkennung();

        assertEquals('n', kennung.getStatus());
        assertEquals('h', kennung.getTyp());
    }

    @Test
    public void testGetId() {
        String satzId = "TLB0123456";

        MabFeld feld = new MabFeld(1, ' ');
        feld.setInhalt(satzId);

        satz.addFeld(feld);

        assertEquals(satzId, satz.getId());
    }

    @Test
    public void testGetIdWithoutMAB001() {
        assertNull(satz.getId());
    }

    @Test
    public void testModifyId() {
        String satzId = "ID0001";

        MabFeld feld = new MabFeld(1, ' ');
        feld.setInhalt(satzId);

        satz.addFeld(feld);

        assertEquals(satzId, satz.getId());

        satzId = "ID0002";

        List<IMabFeld> felder = satz.getFelder(MabFeldDefinition.MAB001);

        felder.get(0).setInhalt(satzId);

        assertEquals(satzId, satz.getId());
    }

    @Test
    @Ignore(value="Functionality not implemented yet.")
    public void testSwitchPositionOfTwoFields() {
        String satzId1 = "ID0001";
        String satzId2 = "ID0002";

        MabFeld feld1 = new MabFeld(26, ' ', satzId1);
        MabFeld feld2 = new MabFeld(26, ' ', satzId2);

        satz.addFeld(feld1);
        satz.addFeld(feld2);

        List<IMabFeld> felder = satz.getFelder(MabFeldDefinition.MAB026);

        assertEquals(satzId1, felder.get(0).getInhalt());
        assertEquals(satzId2, felder.get(1).getInhalt());

        // TODO implement functions for performing this operation
    }

    @Test
    public void testAddAlephFeld() throws Exception {
        satz.addFeld(new MabFeld("LOW", ' ', "LIB001"));
        satz.addFeld(new MabFeld("LOW", ' ', "LIB002"));

        List<IMabFeld> felder = satz.getFelder("LOW ");

        assertEquals(2, felder.size());
        assertEquals("LIB001", felder.get(0).getInhalt());
        assertEquals("LIB002", felder.get(1).getInhalt());
    }

    @Test
    public void testGetFelderForMabFeldDefinition() {
        String satzId1 = "ID0001";
        String satzId2 = "ID0002";

        MabFeld feld1 = new MabFeld(26, ' ', satzId1);
        MabFeld feld2 = new MabFeld(26, ' ', satzId2);

        satz.addFeld(feld1);
        satz.addFeld(feld2);

        List<IMabFeld> felder = satz.getFelder(MabFeldDefinition.MAB026);

        assertEquals(2, felder.size());
        assertEquals(satzId1, felder.get(0).getInhalt());
        assertEquals(satzId2, felder.get(1).getInhalt());
    }

    @Test
    public void testGetFelderForCategory() {
        String satzId1 = "ID0001";
        String satzId2 = "ID0002";

        MabFeld feld1 = new MabFeld(26, ' ', satzId1);
        MabFeld feld2 = new MabFeld(26, 'a', satzId2);

        satz.addFeld(feld1);
        satz.addFeld(feld2);

        List<IMabFeld> felder = satz.getFelder("026");

        assertEquals(2, felder.size());
        assertEquals(satzId1, felder.get(0).getInhalt());
        assertEquals(satzId2, felder.get(1).getInhalt());
    }

    @Test
    public void testGetFelderForCategoryAndIndikator() {
        satz.addFeld(new MabFeld(540, ' ', "value"));
        satz.addFeld(new MabFeld(540, 'a', "value2"));
        satz.addFeld(new MabFeld(540, 'b', "value3"));

        List<IMabFeld> felder = satz.getFelder("540", 'a');

        assertEquals(1, felder.size());
        assertEquals("value2", felder.get(0).getInhalt());
    }

    @Test
    public void testGetFelderForCategoryAndZeroIndikator() {
        satz.addFeld(new MabFeld(540, ' ', "value"));
        satz.addFeld(new MabFeld(540, 'a', "value2"));
        satz.addFeld(new MabFeld(540, 'b', "value3"));

        List<IMabFeld> felder = satz.getFelder("540", ( char )0);

        assertEquals(3, felder.size());
    }

    @Test
    @Ignore(value="feature not implemented yet")
    public void testSortingOfCategories() {
    }

}
