/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.util.ArrayList;
import java.util.List;
import static org.junit.Assert.*;

import org.junit.Ignore;
import org.junit.Test;

/**
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class MabFeldTest {

    @Test
    public void testCreatingFeldInstance() {
        MabFeld feld = new MabFeld(331, 'i');
        assertEquals(331, feld.getFeldnummer());
        assertEquals('i', feld.getIndikator());
        assertNull(feld.getInhalt());
        assertNull(feld.getUnterfelder());
    }

    @Test
    public void testCreatingFeldInstanceWithContent() {
        String content = "Title";
        MabFeld feld = new MabFeld(331, 'i', content);
        assertEquals(331, feld.getFeldnummer());
        assertEquals('i', feld.getIndikator());
        assertEquals(content, feld.getInhalt());
        assertNull(feld.getUnterfelder());
    }

    @Test
    public void testCreatingFeldInstanceWithSubfields() {
        List<IMabUnterfeld> subfields = new ArrayList<IMabUnterfeld>();
        subfields.add(new MabUnterfeld('a', "test"));
        MabFeld feld = new MabFeld(331, 'i', subfields);
        assertEquals(331, feld.getFeldnummer());
        assertEquals('i', feld.getIndikator());
        assertTrue(feld.hatUnterfelder());
        assertNotNull(feld.getUnterfelder());
    }

    @Test
    public void testCreatingFeldInstanceWithSubfieldAndContent() {
        String content = "title";
        List<IMabUnterfeld> subfields = new ArrayList<IMabUnterfeld>();
        subfields.add(new MabUnterfeld('a', "test"));
        MabFeld feld = new MabFeld(331, 'i', content, subfields);
        assertEquals(331, feld.getFeldnummer());
        assertEquals('i', feld.getIndikator());
        assertTrue(feld.hatUnterfelder());
        assertNotNull(feld.getUnterfelder());
        assertEquals(content, feld.getInhalt());
    }

    @Test
    public void testCreatingNonMab2Feld() throws Exception {
        MabFeld feld = new MabFeld("LOW", ' ');
        assertEquals("LOW ", feld.getBezeichnung());
        assertEquals(' ', feld.getIndikator());
    }

    @Test
    public void testCreatingNonMab2FeldWithContent() throws Exception {
        MabFeld feld = new MabFeld("LOW", ' ', "LIB001");
        assertEquals("LOW ", feld.getBezeichnung());
        assertEquals(' ', feld.getIndikator());
        assertEquals("LIB001", feld.getInhalt());
    }

    @Test(expected=MabException.class)
    public void testCreatingFeldCategoryTooShort() throws Exception {
        MabFeld feld = new MabFeld("26", ' ');
    }

    @Test(expected=MabException.class)
    public void testCreatingFeldCategoryNull() throws Exception {
        MabFeld feld = new MabFeld(null, ' ');
    }

    @Test(expected=MabException.class)
    public void testCreatingFeldWithContentCategoryTooLong() throws Exception {
        MabFeld feld = new MabFeld("LOWER", ' ', "content");
    }

    @Test(expected=MabException.class)
    public void testCreatingFeldWithContentCategoryNull() throws Exception {
        MabFeld feld = new MabFeld(null, ' ', "content");
    }

    @Test
    @Ignore
    public void testFeldnummerString() {
        MabFeld feld = new MabFeld(331, 'i');
        // assertEquals(331, feld.getFeldnummerString());
    }

    @Test
    @Ignore
    public void testFeldnummerStringWithLeadingZeros() {
        MabFeld feld = new MabFeld(1,' ');
        // assertEquals("001", feld.getFeldnummerString());
    }

    @Test
    public void testGetBezeichnung() {
        MabFeld feld = new MabFeld(331, 'i');
        assertEquals("331i", feld.getBezeichnung());
    }

    @Test
    public void testGetBezeichnungWithLeadingZeros() {
        MabFeld feld = new MabFeld(52, 'a');
        assertEquals("052a", feld.getBezeichnung());
    }

    @Test
    public void testAddSubfield() {
        MabFeld feld = new MabFeld(52, 'a');
        assertFalse(feld.hatUnterfelder());
        assertNull(feld.getUnterfelder());
        feld.addSubfield(new MabUnterfeld('b', "test"));
        assertTrue(feld.hatUnterfelder());
        assertNotNull(feld.getUnterfelder());
        assertEquals(1, feld.getUnterfelder().size());
    }

    @Test
    public void testAddAdditionalSubfield() {
        List<IMabUnterfeld> subfields = new ArrayList<IMabUnterfeld>();
        subfields.add(new MabUnterfeld('a', "test"));
        MabFeld feld = new MabFeld(331, 'i', subfields);
        assertEquals(331, feld.getFeldnummer());
        assertEquals('i', feld.getIndikator());
        assertTrue(feld.hatUnterfelder());
        assertNotNull(feld.getUnterfelder());
        assertEquals(1, feld.getUnterfelder().size());
        feld.addSubfield(new MabUnterfeld('b', "test2"));
        assertEquals(2, feld.getUnterfelder().size());
    }

    @Test
    public void testRemoveSubfield() {
        MabFeld feld = new MabFeld(331, 'i');

        feld.addSubfield(new MabUnterfeld('a', "u1"));
        feld.addSubfield(new MabUnterfeld('b', "u2"));

        List<IMabUnterfeld> unterfelder = feld.getUnterfelder();

        feld.removeSubfield(unterfelder.get(0));

        unterfelder = feld.getUnterfelder();

        assertEquals(1, unterfelder.size());
    }

    @Test
    public void testRemoveSubfieldThatIsNotPresent() {
        MabFeld feld = new MabFeld(331, 'i');

        feld.removeSubfield(new MabUnterfeld('a', "u1"));
    }

    @Test
    @Ignore(value="Sollte die Liste kopiert werden?")
    public void testRemoveSubfieldWithoutSideEffects() {
        MabFeld feld = new MabFeld(331, 'i');

        feld.addSubfield(new MabUnterfeld('a', "u1"));
        feld.addSubfield(new MabUnterfeld('b', "u2"));

        List<IMabUnterfeld> unterfelder = feld.getUnterfelder();

        feld.removeSubfield(unterfelder.get(0));

        assertEquals(2, unterfelder.size());
    }

    @Test
    public void testHatUnterfelderForFieldWithoutSubfields() {
        MabFeld feld = new MabFeld(52, 'a');
        assertFalse(feld.hatUnterfelder());
    }

    @Test
    public void testHatInhaltForFieldWithoutContent() {
        MabFeld feld = new MabFeld(52, 'a');
        assertFalse(feld.hatInhalt());
    }

    @Test
    public void testHatInhaltForZeroLengthContent() {
        MabFeld feld = new MabFeld(331, ' ', "");
        assertFalse(feld.hatInhalt());
    }

    @Test
    public void testSetInhalt() {
        String content = "Test Title";
        MabFeld feld = new MabFeld(331, ' ');
        feld.setInhalt(content);
        assertEquals(content, feld.getInhalt());
    }

    @Test
    public void testSetUnterfelderToNull() {
        MabFeld feld = new MabFeld(331, ' ');
        feld.setUnterfelder(null);
        assertFalse(feld.hatUnterfelder());
        assertNull(feld.getUnterfelder());
    }

    @Test
    public void testSetUnterfelderWithEmptyList() {
        MabFeld feld = new MabFeld(331, ' ');
        feld.setUnterfelder(new ArrayList<IMabUnterfeld>());
        assertFalse(feld.hatUnterfelder());
        assertNull(feld.getUnterfelder());
    }

    @Test
    public void testGetUnterfelder() {
        MabFeld feld = new MabFeld(331, ' ');
        feld.addSubfield(new MabUnterfeld('a', "u1"));
        feld.addSubfield(new MabUnterfeld('b', "u2"));

        List<IMabUnterfeld> unterfelder = feld.getUnterfelder();

        assertEquals(2, unterfelder.size());
        assertEquals('a', unterfelder.get(0).getKennzeichen());
        assertEquals('b', unterfelder.get(1).getKennzeichen());
    }

    @Test
    public void testToString() {
        MabFeld feld = new MabFeld(52, 'a');
        String output = feld.toString();
        assertNotNull(output);
    }

    @Test
    public void testToStringWithContent() {
        MabFeld feld = new MabFeld(331, ' ', "Test Title");
        String output = feld.toString();
        assertNotNull(output);
    }

    @Test
    public void testToStringWithUnterfeldern() {
        List<IMabUnterfeld> subfields = new ArrayList<IMabUnterfeld>();
        subfields.add(new MabUnterfeld('a', "test"));
        MabFeld feld = new MabFeld(331, 'i', subfields);
        String output = feld.toString();
        assertNotNull(output);
    }

    @Test
    public void testGetFeldnummerForNonMab2Feld() throws Exception {
        MabFeld feld = new MabFeld("LOW", ' ', "content");

        assertEquals(-1, feld.getFeldnummer());
    }

}
