/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.io.*;
import java.net.URL;
import java.util.EnumSet;
import java.util.Set;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.apache.commons.configuration.XMLConfiguration;
import org.apache.commons.configuration.tree.xpath.XPathExpressionEngine;
import org.apache.commons.lang.StringUtils;
import static org.junit.Assert.*;

import org.junit.Ignore;
import org.junit.Test;

/**
 * Unit tests for MabFeldDefinition enum.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class MabFeldDefinitionTest {

    @Test
    public void isMabField() {
        assertTrue(MabFeldDefinition.isMabField(1));
    }

    @Test
    public void isMabFieldUnknownNumber() {
        assertFalse(MabFeldDefinition.isMabField(7));
    }

    @Test
    public void testGetFeld() {
        MabFeldDefinition feld = MabFeldDefinition.getFeld(1);
        assertEquals(MabFeldDefinition.MAB001, feld);
    }

    @Test
    public void testAttemptToGetUnknownFeld() {
        MabFeldDefinition feld = MabFeldDefinition.getFeld(7);
        assertNull(feld);
    }

    @Test
    public void testGetFeldnummerString() {
        assertEquals("001", MabFeldDefinition.getFeldnummerString(1));
        assertEquals("081", MabFeldDefinition.getFeldnummerString(81));
        assertEquals("200", MabFeldDefinition.getFeldnummerString(200));
    }

    @Test
    public void testGetWiederholfaktor() {
        MabFeldDefinition feld = MabFeldDefinition.getFeld(596);
        assertEquals(MabFeldDefinition.MAB596, feld);
        assertEquals(20, feld.getWiederholfaktor());
    }

    @Test
    public void testGetWiederholFaktorForFieldWithDefaultValue() {
        MabFeldDefinition feld = MabFeldDefinition.getFeld(1);
        assertEquals(MabFeldDefinition.MAB001, feld);
        assertEquals(1, feld.getWiederholfaktor());
    }

    @Test
    public void testGetFeldName() {
        MabFeldDefinition feld = MabFeldDefinition.getFeld(592);
        assertEquals(MabFeldDefinition.MAB592, feld);
        assertEquals("Abteilung/Unterreihe der Quelle", feld.getName());
    }

    @Test
    public void testNoMissingFieldNames() {
        for (MabFeldDefinition feld : EnumSet.allOf(MabFeldDefinition.class)) {
            assertFalse("Name of field " + feld.getFeldnummerString()
                    + " is missing.", StringUtils.isBlank(feld.getName()));
        }
    }

    @Test
    public void testIsMab2Field() {
        assertTrue(MabFeldDefinition.isMab2Field(1, ' ', false));
    }

    @Test
    public void testIsMab2FieldBadIndicator() {
        assertFalse(MabFeldDefinition.isMab2Field(1, 'a', false));
    }

    @Test
    public void testIsMab2FieldUnknownNumber() {
        assertFalse(MabFeldDefinition.isMab2Field(7, ' ', false));
    }

    /**
     * TODO verify expectation (true or false)?
     */
    @Test
    public void testIsMab2FieldWithUnterfeldzwang() {
        assertFalse(MabFeldDefinition.isMab2Field(705, ' ', true));
    }

    @Test
    public void testIsSubfieldRequiredTrue() {
        assertTrue(MabFeldDefinition.MAB705.isSubfieldRequired());
    }

    @Test
    public void testIsSubfieldRequiredFalse() {
        assertFalse(MabFeldDefinition.MAB001.isSubfieldRequired());
    }

    @Test
    public void testGetUnterfeldkennzeichen() {
        assertNotNull(MabFeldDefinition.MAB705.getUnterfeldkennzeichen());
    }

    @Test
    public void testGetUnterfeldkennzeichenForFieldWithoutSubfields() {
        assertEquals(0, MabFeldDefinition.MAB001.getUnterfeldkennzeichen().length);
    }

    @Test
    public void testIsIndikatorWiederholbarFalse() {
        assertFalse(MabFeldDefinition.MAB025.isIndikatorWiederholbar());
    }

    @Test
    public void testIsIndikatorWiederholbarTrue() {
        assertTrue(MabFeldDefinition.MAB201.isIndikatorWiederholbar());
    }

    @Test
    public void testGetIndikatoren() {
        MabFeldDefinition feld = MabFeldDefinition.MAB025;
        assertNotNull(feld.getIndikatoren());
        assertEquals(11, feld.getIndikatoren().length);
    }

    @Test
    public void testHatKennzeichenTrue() {
        MabFeldDefinition feld = MabFeldDefinition.MAB705;
        assertTrue(feld.hatKennzeichen('a'));
    }

    @Test
    public void testHatKennzeichenFalse() {
        MabFeldDefinition feld = MabFeldDefinition.MAB705;
        assertFalse(feld.hatKennzeichen('o'));
    }

    @Test
    public void testHatKennzeichenFalseForFieldWithoutSubfields() {
        MabFeldDefinition feld = MabFeldDefinition.MAB001;
        assertFalse(feld.hatKennzeichen('a'));
    }

    @Test
    public void testToStringForAll() {
        for (MabFeldDefinition feld : EnumSet.allOf(MabFeldDefinition.class)) {
            assertNotNull(feld.toString());
        }
    }

    @Test
    public void testGetWiederholfaktorForAll() {
        for (MabFeldDefinition feld : EnumSet.allOf(MabFeldDefinition.class)) {
            feld.getWiederholfaktor();
        }
    }

    @Test
    @Ignore
    public void testIsRequiredForAll() {
        for (MabFeldDefinition feld : EnumSet.allOf(MabFeldDefinition.class)) {
            // feld.isRequired();
        }
    }

    @Test
    public void testReadMabInfo() throws IOException, ConfigurationException {
        URL mabInfoUrl = getClass().getClassLoader().getResource(
                "de/kobv/mable/mab/mabinfo.xml");

        HierarchicalConfiguration config = new XMLConfiguration(mabInfoUrl);

        config.setExpressionEngine(new XPathExpressionEngine());

        assertEquals("blank", config.getString(
                "mabfield[@id='default']/indikatoren"));
        assertEquals("true", config.getString(
                "mabfield[@id='001']/pflichtfeld"));
        assertTrue(config.getBoolean(
                "mabfield[@id='001']/pflichtfeld"));
    }

    @Test
    public void testIsMabSubfieldTrue() {
        assertTrue(MabFeldDefinition.isMabSubfield(418, ' ', 'a'));
    }

    @Test
    public void testIsMabSubfieldFalse() {
        assertFalse(MabFeldDefinition.isMabSubfield(418, ' ', 'i'));
    }

    @Test
    public void testIsMabSubfieldForBadCategory() {
        assertFalse(MabFeldDefinition.isMabSubfield(419, ' ', 'i'));
    }

    @Test
    public void testIsMabSubfieldForBadIndikator() {
        assertFalse(MabFeldDefinition.isMabSubfield(418, 'd', 'i'));
    }

    @Test
    public void testGetSatztypen() {
        Set<String> satztypen = MabFeldDefinition.MAB010.getSatztypen();

        assertEquals(4, satztypen.size());
        assertTrue(satztypen.contains("y"));
        assertTrue(satztypen.contains("u"));
        assertTrue(satztypen.contains("l"));
        assertTrue(satztypen.contains("e"));
    }

    @Test
    public void testGetSatztypenForFieldWithoutRestrictions() {
        Set<String> satztypen = MabFeldDefinition.MAB001.getSatztypen();

        assertNull(satztypen);
    }

    @Test
    public void testGetSatztypenForFieldWithOneType() {
        Set<String> satztypen = MabFeldDefinition.MAB031.getSatztypen();

        assertEquals(1, satztypen.size());
        assertTrue(satztypen.contains("c"));
    }

    @Test
    public void testIsRequiredForIgnore() {
        assertFalse(MabFeldDefinition.MAB015.isRequired());
    }

    @Test
    public void testIsRequiredForTrue() {
        MabFeldDefinition.MAB001.isRequired();
        assertTrue(MabFeldDefinition.MAB001.isRequired());
    }

    @Test
    public void testIsRequiredFalse() {
        assertFalse(MabFeldDefinition.MAB005.isRequired());
    }

    @Test
    public void testIsRequiredForSatztypen() {
        assertFalse(MabFeldDefinition.MAB331.isRequired());
    }

    @Test
    public void testIsRequiredForSatztyp() {
        assertTrue(MabFeldDefinition.MAB331.isRequiredForSatztyp("h"));
    }

    @Test
    @Ignore
    public void testMAB070RequiredForIndikatorA() {
        // TODO MAB070a ist Pflichtfeld während 070_ und 070b keine Pflichtfelder sind
    }

    @Test
    public void testCaching() {
        MabFeldDefinition feldDef1 = MabFeldDefinition.MAB100;
        assertFalse(feldDef1.isRequiredLoaded());
        assertFalse(feldDef1.isRequired());
        assertTrue(feldDef1.isRequiredLoaded());
        MabFeldDefinition feldDef2 = MabFeldDefinition.MAB100;
        assertTrue(feldDef2.isRequiredLoaded());
    }

}
