/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.log4j.Logger;

/**
 * Reads errors report and filters by dataset identifier or error code.
 * @author Jens Schwidder <schwidder(at)zib.de>
 * TODO base implementation on ErrorReportParser?
 * TODO move to specific package
 */
public class ErrorsReportFilter {

    /**
     * Logger for this class.
     */
    private static final Logger LOG = Logger.getLogger(
            ErrorsReportFilter.class);

    /**
     * Error code for filtering.
     */
    private int filterErrorCode;

    /**
     * Dataset identifier for filtering.
     */
    private String filterDatasetId;

    /**
     * Reader for report data.
     */
    private Reader in;

    /**
     * Filters report content for lines matching dataset ID or error code.
     *
     * If dataset ID and error code are set, the function will only filter by
     * dataset ID.
     *
     * @return Set<String> Lines containing dataset ID and error code
     * @throws IOException if an error occurs while reading the date
     */
    public Set<String> filter() throws IOException {
        BufferedReader reader = new BufferedReader(in);

        String patternStr;

        if (filterDatasetId != null) {
            patternStr = String.format("(%1$s), #(\\d+)", filterDatasetId);
        }
        else {
            patternStr = String.format("(.+), #(%1$4d)", filterErrorCode);
        }

        Pattern pattern = Pattern.compile(patternStr);

        Set<String> result = new HashSet<String>();

        String line;

        while ((line = reader.readLine()) != null) {
            Matcher matcher = pattern.matcher(line);
            if (matcher.find()) {
                String datasetId = matcher.group(1);
                String errorCode = matcher.group(2);
                if (this.filterDatasetId != null) {
                    result.add(errorCode);
                }
                else {
                    result.add(datasetId);
                }
            }
        }

        return result;
    }

    /**
     * Getter for dataset ID (MAB001) for filtering.
     * @return String MAB001 value
     */
    public String getDatasetId() {
        return filterDatasetId;
    }

    /**
     * Setter for dataset ID (MAB001) for filtering.
     * @param datasetId String MAB001
     */
    public void setDatasetId(final String datasetId) {
        this.filterDatasetId = datasetId;
    }

    /**
     * Getter for error code for filtering.
     * @return int Error code
     */
    public int getErrorCode() {
        return filterErrorCode;
    }

    /**
     * Setter for error code for filtering.
     * @param errorCode int Error code from report
     */
    public void setErrorCode(final int errorCode) {
        this.filterErrorCode = errorCode;
    }

    /**
     * Getter for reader for report content.
     * @return Reader for report content
     */
    public Reader getSource() {
        return in;
    }

    /**
     * Setter for reader for report content.
     * @param source Reader for report content
     */
    public void setSource(final Reader source) {
        this.in = source;
    }

}
