/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.xml;

import de.ddb.charset.MabCharset;
import de.kobv.mable.common.MableConfigurationException;
import de.kobv.mable.mab.MabSatzkennung;
import de.kobv.mable.mab.parser.ContentHandler;
import org.apache.log4j.Logger;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAXParser Handler for processing MAB XML data.
 *
 * This parser passes information immediately to content handler for processing.
 * 
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
public class MabXmlHandler extends DefaultHandler {

    /**
     * Logger for this class.
     */
    private static final Logger LOG = Logger.getLogger(MabXmlHandler.class);

    /**
     * TODO Mable specific; remove it?
     */
    private ContentHandler contentHandler;

    private String datasetId;

    private String datasetType;

    private String datasetStatus;

    private String feldnummer;

    private String indikator;

    private String unterfeldkennung;

    /**
     * Content of dataset.
     */
    private StringBuilder content = new StringBuilder();

    private int datasetCount;

    /**
     * Possible states while parsing MABXML.
     */
    private enum State {

        NONE,

        IN_DOCUMENT,

        IN_DATASET,

        IN_FELD,

        IN_UNTERFELD;

    }

    private State status;

    @Override
    public void characters(final char[] ch, final int start,
            final int length) throws SAXException {
        LOG.debug("characters: " + new String(ch, start, length));
        content.append(ch, start, length);
    }

    @Override
    public void endDocument() throws SAXException {
        LOG.debug("endDocument");
        contentHandler.endParsing(); // TODO mable specific
        status = State.NONE;
    }

    @Override
    public void endElement(final String uri, final String localName,
            final String qName) throws SAXException {
        LOG.debug("endElement " + qName);
        if ("datensatz".equals(qName)) {
            contentHandler.endSatz();
            status = State.IN_DOCUMENT;
            datasetCount++;
        }
        else if ("feld".equals(qName)) {
            LOG.debug("field " + feldnummer + indikator + " = '" + content.toString() + "'");
            contentHandler.startFeld(feldnummer, indikator.charAt(0),
                    content.toString());
            status = State.IN_DATASET;
        }
        else if ("ns".equals(qName)) {
            content.append(MabCharset.NICHTSORTIERENDEZEICHEN);
        }
        else if ("stw".equals(qName)) {
            content.append(MabCharset.STICHWORTENDEZEICHEN);
        }
        else if ("uf".equals(qName)) {
            contentHandler.startUnterfeld(unterfeldkennung.charAt(0),
                    content.toString());
            status = State.IN_FELD;
        }
    }

    @Override
    public void startDocument() throws SAXException {
        LOG.debug("startDocument");
        try {
            contentHandler.startParsing();
        }
        catch (MableConfigurationException mce) {
            LOG.error(mce);
            throw new SAXException("Error while starting parsing.", mce);
        }
        status = State.IN_DOCUMENT;
    }

    @Override
    public void startElement(final String uri, final String localName,
            final String qName, final Attributes attributes)
            throws SAXException {
        LOG.debug("startElement " + qName);
        if ("datensatz".equals(qName)) {
            this.datasetId = attributes.getValue("id");
            LOG.debug("Parsing dataset id = " + datasetId);
            this.datasetType = attributes.getValue("typ");
            this.datasetStatus = attributes.getValue("status");
            MabSatzkennung satzkennung = new MabSatzkennung(0, datasetType,
                    datasetStatus);
            contentHandler.startSatz(satzkennung);
        }
        else if ("feld".equals(qName)) {
            this.feldnummer = attributes.getValue("nr");
            this.indikator = attributes.getValue("ind");
            status = State.IN_FELD;
            content.setLength(0);
        }
        else if ("tf".equals(qName)) {
            content.append(MabCharset.TEILFELDTRENNZEICHEN);
        }
        else if ("ns".equals(qName)) {
            content.append(MabCharset.NICHTSORTIERBEGINNZEICHEN);
        }
        else if ("stw".equals(qName)) {
            content.append(MabCharset.STICHWORTBEGINNZEICHEN);
        }
        else if ("uf".equals(qName)) {
            unterfeldkennung = attributes.getValue("code");
            status = State.IN_UNTERFELD;
            content.setLength(0);
        }
    }

    public ContentHandler getContentHandler() {
        return contentHandler;
    }

    public void setContentHandler(ContentHandler handler) {
        this.contentHandler = handler;
    }

    public int getDatasetsProcessed() {
        return datasetCount;
    }

}
