/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.parser;

import de.ddb.charset.MabCharset;
import de.kobv.mable.ErrorCode;
import de.kobv.mable.common.MableConfigurationException;
import de.kobv.mable.mab.MabFeldDefinition;
import de.kobv.mable.mab.MabSatzkennung;
import de.kobv.mable.mab.validation.MabCharsetChecker;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.text.ParseException;
import java.util.LinkedList;
import java.util.Queue;

/**
 * Ein Parser zum Einlesen von MAB 2.0 Banddateien, der vor allem auf Stabilität
 * ausgelegt ist. Der Parser verfügt über eine Statemachine und untersucht die
 * Banddatei zeichenweise. Wird ein Satz, ein Feld oder ein Unterfeld mit
 * so gravierenden Mängeln gefunden, dass es
 * nicht in die Datenbank eingelesen werden kann, wird es in eine Reject-Datei
 * geschrieben. In der Datenbank wird dann vermerkt, dass die Datei nicht
 * vollständig eingelesen werden konnte. Beim Einlesen werden bereits Checks auf
 * Validität der Banddatei durchgeführt. Fehler werden in der Datenbank
 * vermerkt.
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO get rid of ParseException or replace will appropriate class
 * TODO add method switchState(NEW_STATE) for centralizing steps that need to be taken like resetFeld();
 * TODO fehlerPuffer sinnvoll? (nur bei Datenbank, oder?) evtl. auch um Fehler zu "Positionieren"
 *
 */
public class DefaultMabParser implements MabParser {

    /**
     * Size of read buffer.
     */
    private static final int BUFFER_SIZE = 8 * 1024;

    /**
     * Length of MAB dataset header.
     */
    private static final int HEADER_LENGTH = 24;

    /**
     * Zustände der Statemachine des {@link DefaultMabParser}.
     * @author Pascal-Nicolas Becker <becker(at)zib.de>
     *
     */
    private static enum Zustand {

        /**
         * Undefinierter Zustand. Nach der Initilisierung des Parsers wird
         * dieser Zustand eingenommen. Das Aufrufen der von
         * {@code DefaultMabParser#parse()} versetzt uns dann zu nächst in den Zustand
         * {@code #IN_DOKUMENT}.
         */
        UNDEFINIERT,

        /**
         * Zustand wird eingenommen wenn als nächstes eine Satzkennung erwartet
         * wird, also nach dem öffnen einer Datei oder nach jedem vollständig
         * gelesenen Satz.
         */
        IN_DOKUMENT,

        /**
         * Beschreibt, dass die Statemachine sich in einem Satz aber ausserhalb
         * eines Feldes befindet. Als nächstes wird also eine Feldnummer
         * erwartet.
         */
        IN_SATZ,

        /**
         * Der Parser befindet sich in einem Satz, der so defekt war, dass er
         * nicht in die Datenbank aufgenommen werden kann, daher in die
         * Reject-Datei geschrieben wird.
         */
        IGNORIERE_SATZ,

        /**
         * Feldnummer und Indikator wurden gelesen, der Parser unterucht den
         * Inhalt eines Feldes.
         *
         */
        IN_FELD,

        /**
         * Ein Unterfeldbeginnzeichen wurde erkannt, nun wird das Kennzeichen
         * erwartet.
         */
        NACH_UNTERFELDSTART,

        /**
         * Wir befinden uns in einem Unterfeld und lesen den Inhalt des Feldes
         * aus, bis zum Auftreten eines Mab-Kontrollzeichens (wie z.B. eines
         * Feldendezeichens).
         */
        IN_UNTERFELD
    }

    /** Bytewert eines UTF-8-kodierten Carriage return. */
    public static final int CARRIAGERETURN = 0x000D;

    /** Bytewert eines UTF-8-kodierten Zeilenumbruchs. */
    public static final int LINEFEED = 0x000A;

    /** Reader, der die MAB-Banddatei einliest. */
    private Reader in;

    /** MABle-Verbindung zur Datenbank. */
    private ContentHandler contentHandler;

    private ErrorHandler errorHandler;

    /** Speichert den aktuellen Zustand der Statemachine. */
    private Zustand zustand;

    /** Zählt die gelesenen Zeichen. */
    private int position;

    /** Puffer zum Zwischenspeichern von Satzkennung, Feldinhalten oder
     * ähnlichem. */
    private StringBuilder puffer;

    /** Puffer, falls Zeichen zurück gelegt oder eingefügt werden sollen (wenn
     * z.B. ein Unterfeld mit einem Satzendezeichen beendet wird, wird dies in
     * der Datenbank als Fehler gespeichert, der Parser fügt zur internetn
     * Verarbeitung ein Feldendezeichen ein, legt das Satzendezeiechen zurück
     * und liest beides erneut).*/
    private Queue<Character> toInsert;

    /** Speichert Fehler zwischen, falls eine Einheit erst vollständig
     * eingelesen werden muss, bis die Fehler in der Datenbank gespeichert
     * werden können.
     */
    private Queue<ErrorCode> fehlerPuffer;

    /** Gibt an, dass Daten ignoriert und in die Reject-Datei geschrieben
     * wurden. */
    private boolean datenIgnoriert;

    /** Reason for ignoring dataset. */
    private ErrorCode ignoredDatasetError;

    /** MAB001 of ignored dataset. */
    private String ignoredDatasetId;

    /** Reason for ignoring field. */
//    private ErrorCode ignoredFieldError; // TODO not used

    /** Speichert, ob der Parser sich in einem Nichtsortierbereich befindet. */
    private boolean inNichtsortiert;

    /** Speichert, ob der Parser sich in einem Stichwortbereich befindet. */
    private boolean inStichwort;

    /**
     * Speichert das Satzinformationen in die Reject-Datei geschrieben werden
     * sollen.
     *
     * TODO is used for ignored data (not printing) -> rename
     */
    private boolean printSatzInfos;

    /** Speichert die Anzahl der gefundenen Sätze. */
    private int saetzeGefunden;

    /** Speichert die Anzahl der ignorierten Sätze. */
    private int saetzeIgnoriert;

    /** Speichert die Anzahl der ignorierten Felder. */
//    private int felderIgnoriert; // TODO not used

    /** Speichert die Anzahl der ignorierten Unterfelder. */
//    private int unterfelderIgnoriert; // TODO not used

    /** String-Repräsentation von {@code #iFeldnummer} wie sie aus der MAB-Datei
     * ausgelesen wurde. */
    private String sFeldnummer;

    /**
     * Speichert die Feldnummer nach der Erkennung
     * ({@link java.lang.Integer#parseInt(String)} zwischen, da erste das
     * gesamte Feld eingelesen werden muss, ehe es eingetragen werden kann,
     * beziehungsweise um Unterfeldkennzeichen zu überprüfen.
     */
    private int iFeldnummer;

    /** Speichert den Feldindikator zwischen.*/
    private char feldindikator;

    /** Speichert ein Unterfeldkennzeichen zwischen. */
    private char unterfeldkennzeichen;

    /**
     * Used to signal first field for ignored datasets.
     */
    private boolean checkFirstField = false;

    /**
     * Initialisiert den DefaultMabParser.
     */
    public DefaultMabParser()  {
        // intialisieren der Variablen
    }

    /**
     * Setter for MAB data handler.
     * @param handler ContentHandler
     */
    public void setContentHandler(final ContentHandler handler) {
        this.contentHandler = handler;
    }

    /**
     * Setter for MAB parser error handler.
     * @param handler Handler for errors while parsing.
     */
    public void setErrorHandler(final ErrorHandler handler) {
        this.errorHandler = handler;
    }

    /**
     * Setter for input Reader.
     * @param source Input source for MAB data.
     */
    @Override
    public void setSource(final Reader source) {
        if (source instanceof BufferedReader) {
            this.in = source;
        }
        else {
            this.in = new BufferedReader(source, BUFFER_SIZE);
        }
    }

    /**
     * Initializes parser.
     */
    public void init() {
        this.zustand = Zustand.UNDEFINIERT;
        this.position = 0;
        this.puffer = new StringBuilder();
        this.fehlerPuffer = new LinkedList<ErrorCode>();
        this.datenIgnoriert = false;
        this.toInsert = new LinkedList<Character>();
        this.inNichtsortiert = false;
        this.inStichwort = false;
        this.printSatzInfos = false;

        this.saetzeGefunden = 0;
        this.saetzeIgnoriert = 0;
//        this.felderIgnoriert = 0;
//        this.unterfelderIgnoriert = 0;

        this.iFeldnummer = -1;
        this.sFeldnummer = "";
        this.unterfeldkennzeichen = '#';
    }

    /**
     * Parsed die MAB-Datei (Hauptroutine des MabParsers).
     * @throws IOException falls beim Lesen/Schreiben eine IOException auftritt
     * @throws ParseException falls ein schwerer interner Fehler auftritt, in
     * diesem Fall bitten den Entwickler beanchrichtigen.
     */
    @Override
    public void parse() throws IOException, ParseException, MableConfigurationException {
        init();
        this.contentHandler.startParsing();
        this.zustand = Zustand.IN_DOKUMENT;

        int input;

        // Schleife solange Tokens gelesen werden (also bis EOF oder IOException
        while ((input = this.nextToken()) != -1) {
            // caste int in character
            char inputChar = (char) input;

            // Statemachine: behandle Token je nach Zustand
            switch (this.zustand) {
                case IN_DOKUMENT:
                    parseInDocument(inputChar);
                    break;
                case IN_SATZ:
                    parseInSatz(inputChar);
                    break;
                case IN_FELD:
                    parseInFeld(inputChar);
                    break;
                case NACH_UNTERFELDSTART:
                    parseNachUnterfeldStart(inputChar);
                    break;
                case IN_UNTERFELD:
                    parseInUnterfeld(inputChar);
                    break;
                case IGNORIERE_SATZ:
                    parseIgnoredDataset(inputChar);
                    break;
                case UNDEFINIERT:
                    // Kann nicht auftreten, da die switch-Anweisung
                    // nur IN_DOKUMENT erreicht wird und von dort aus
                    // nicht mehr nach UNDEFINIERT ueberfuehrt.
                default:
                    throw new ParseException("undefinierten Zustand erreicht. "
                            + "Bitte den Entwickler benachrichtigen.",
                            this.position);
            } // switch (this.zustand)
        } // while (nextToken)

        finishParsing();
    }

    protected void parseInDocument(final char inputChar) throws IOException {
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                // Feldendezeichen ausserhalb eines Satzes:
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.FELDENDE_OHNE_SATZ);
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
                // Nichtsortierbeginnzeichen ausserhalb eines Satzes
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.NICHTSORTIERBEGINN_OHNE_SATZ);
                break;
            case MabCharset.NICHTSORTIERENDEZEICHEN:
                // Nichtsortierendezeichen ausserhalb eines Satzes
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.NICHTSORTIERENDE_OHNE_SATZ);
                break;
            case MabCharset.SATZENDEZEICHEN:
                // Satzendezeichen ohne Satzbeginn?
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.SATZENDE_OHNE_SATZ);
                break;
            case MabCharset.STICHWORTBEGINNZEICHEN:
                // Stichwortbeginnzeichen ohne Satz?
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.STICHWORTBEGINN_OHNE_SATZ);
                break;
            case MabCharset.STICHWORTENDEZEICHEN:
                // Stichwortendezeichen ausserhalb eines Satzes?
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.STICHWORTENDE_OHNE_SATZ);
                break;
            case MabCharset.TEILFELDTRENNZEICHEN:
                // Teilfeldtrennzeichen ausserhalb eines Satzes?
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.TEILFELDTRENNUNG_OHNE_SATZ);
                break;
            case MabCharset.UNTERFELDBEGINNZEICHEN:
                // Unterfeldbeginnzeichen ausserhalb eines Satzes?
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.UNTERFELDBEGINN_OHNE_SATZ);
                break;
            default:
                parseSatzkennung(inputChar);
                if (this.zustand == Zustand.IGNORIERE_SATZ) {
                    this.checkFirstField = true;
                }
                break;
        } // switch(inputChar)
    }

    protected void parseSatzkennung(final char inputChar) throws IOException {
        // Lese Satzkennung in Puffer ein.
        // TODO refactor? move in separate method?
        this.puffer.append(inputChar);
        boolean eof = false;
        int input;
        for (int i = 0; i < (HEADER_LENGTH - 1) && !eof; i++) {
            input = this.nextToken();
            if (input != -1) {
                this.puffer.append((char) input);
            }
            else {
                eof = true;
            }
        }
        if (eof) {
            // eof wird hier behandelt da break in for
            // schleife nur mit Sprungmarken switch
            // abbrechen koennte
            this.errorHandler.insertLieferungsfehler(
                    ErrorCode.EOF_IN_SATZKENNUNG);
            return;
        }

        // Satzkennung überprüfen
        MabSatzkennung satzkennung = new MabSatzkennung();

        assert(this.puffer.length() == HEADER_LENGTH);

        satzkennung.setValue(this.puffer.substring(0, HEADER_LENGTH));

        // Satzlaenge
        if (!satzkennung.isLengthValid()) {
            this.ignoredDatasetError =
                    ErrorCode.SATZLAENGE_UNZULAESSIG;
            this.zustand = Zustand.IGNORIERE_SATZ;
            return;
        }

        // Satzstatus
        if (!satzkennung.isStatusValid()) {
            this.ignoredDatasetError =
                    ErrorCode.SATZSTATUS_UNZULAESSIG;
            this.zustand = Zustand.IGNORIERE_SATZ;
            return;
        }

        // MAB-Version
        if (!satzkennung.isVersionValid()) {
            this.fehlerPuffer.add(
                    ErrorCode.VERSIONSANGABE_UNZULAESSIG);
        }

        // Indikatorlaenge
        if (!satzkennung.isIndicatorLengthValid()) {
            this.ignoredDatasetError =
                    ErrorCode.INDIKATORLAENGE_UNZULAESSIG;
            this.zustand = Zustand.IGNORIERE_SATZ;
            return;
        }

        //Teilfeldkennungslaenge
        if (!satzkennung.isTeilfeldkennungslaengeValid()) {
            this.fehlerPuffer.add(
                    ErrorCode.TEILFELDKENNUNGSLAENGE_UNZULAESSIG);
        }

        // Datenanfangsadresse
        if (!satzkennung.isDataStartAddressValid()) {
            this.ignoredDatasetError =
                    ErrorCode.DATENANFANGSADRESSE_UNZULAESSIG;
            this.zustand = Zustand.IGNORIERE_SATZ;
            return;
        }

        // nicht benutzt
        if (!satzkennung.isReservedFieldsEmpty()) {
            this.fehlerPuffer.add(
                    ErrorCode.RESERVIERTEZEICHEN_UNZULAESSIG);
        }

        // Satztyp
        if (!satzkennung.isTypeValid()) {
            this.fehlerPuffer.add(ErrorCode.SATZTYP_UNZULAESSIG);
        }

        this.contentHandler.startSatz(satzkennung); // TODO was: this.puffer.toString()

        // Fehler eintragen und dabei Fehlerpuffer leeren
        this.errorHandler.insertSatzfehler(this.fehlerPuffer);
        this.fehlerPuffer.clear();
        this.puffer.setLength(0); // TODO = new StringBuilder();
        // sind nun in einem Satz:
        this.zustand = Zustand.IN_SATZ;
    }

    protected void parseInSatz(final char inputChar) throws IOException {
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                // Feldendezeichen ohne Feldanfang?
                this.errorHandler.insertSatzfehler(ErrorCode.FELDENDE_OHNE_FELD);
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
                // Nichtsortierbeginnzeichen in einem Satz?
                this.errorHandler.insertSatzfehler(
                        ErrorCode.NICHTSORTIERBEGINN_OHNE_FELD);
                break;
            case MabCharset.NICHTSORTIERENDEZEICHEN:
                // Nichtsortierendezeichen ausserhalb von Feld
                this.errorHandler.insertSatzfehler(
                        ErrorCode.NICHTSORTIERENDE_OHNE_FELD);
                break;
            case MabCharset.SATZENDEZEICHEN:
                this.beendeSatz(); // setzt auch Zustand IN_DOKUMENT
                break;
            case MabCharset.STICHWORTBEGINNZEICHEN:
                this.errorHandler.insertSatzfehler(
                        ErrorCode.STICHWORTBEGINN_OHNE_FELD);
                break;
            case MabCharset.STICHWORTENDEZEICHEN:
                this.errorHandler.insertSatzfehler(
                        ErrorCode.STICHWORTENDE_OHNE_FELD);
                break;
            case MabCharset.TEILFELDTRENNZEICHEN:
                this.errorHandler.insertSatzfehler(
                        ErrorCode.TEILFELDTRENNUNG_OHNE_FELD);
                break;
            case MabCharset.UNTERFELDBEGINNZEICHEN:
                this.errorHandler.insertSatzfehler(
                        ErrorCode.UNTERFELDBEGINN_OHNE_FELD);
                break;
            default:
                // lese Feldnummer und Indikator
                // Feldnummer stelle 2 und 3:
                int fnr2 = this.nextToken();
                int fnr3 = this.nextToken();
                if (fnr2 == -1 || fnr3 == -1) {
                    this.errorHandler.insertLieferungsfehler(
                            ErrorCode.EOF_IN_FELDNUMMER);
                    break;
                }
                // Was able to read 3 characters

                String feldnummer = Character.toString(inputChar)
                        + Character.toString((char) fnr2)
                        + Character.toString((char) fnr3);

               this.sFeldnummer = feldnummer;

                // Pruefe Feldnummer
                try {
                    this.iFeldnummer = Integer.parseInt(this.sFeldnummer);
                }
                catch (NumberFormatException nfe) {
                    // muss spaeter eh auf positive Feldnummer checken
                    this.sFeldnummer = "-1";
                    this.iFeldnummer = -1;
                }

                if (this.iFeldnummer <= 0) {
                    // uebergehe dieses Feld
                    this.ignoriereFeld(feldnummer,
                            ErrorCode.FELDNUMMER_FORMATFEHLER);
                    break;
                }

                // Feldinikator
                int iIndikator = this.nextToken();
                if (iIndikator == -1) {
                    this.errorHandler.insertLieferungsfehler(
                            ErrorCode.EOF_IN_FELDINDIKATOR);
                    break;
                }
                this.feldindikator = (char) iIndikator;
                if (!(Character.toString(this.feldindikator
                        ).matches("[a-zA-Z0-9 ]"))) {
                    this.ignoriereFeld(this.sFeldnummer
                            + Character.toString(this.feldindikator),
                            ErrorCode.FELDINDIKATOR_FORMATFEHLER);
                    break;
                }
                this.zustand = Zustand.IN_FELD;
                break;
        } // switch(inputChar)
    }

    protected void parseInFeld(final char inputChar) throws IOException {
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                commonParseCode();
                // pruefe auf Feld-Inhalt:
                if (this.puffer.length() <= 0) {
                    this.fehlerPuffer.add(ErrorCode.LEERES_FELD);
                }

                this.contentHandler.startFeld(this.sFeldnummer,
                        this.feldindikator,
                        this.puffer.toString());

                // Fehler eintragen und dabei Fehlerpuffer leeren.
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator, this.fehlerPuffer);
                // zuruecksetzen
                this.resetFeld();
                // Feld verlassen
                this.zustand = Zustand.IN_SATZ;
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
                if (this.inNichtsortiert) {
                    this.fehlerPuffer.add(
                            ErrorCode.WIEDERHOLTER_NICHTSORTIERBEGINN_IN_FELD);
                }
                this.inNichtsortiert = true;
                this.puffer.append(inputChar);
                break;
            case MabCharset.NICHTSORTIERENDEZEICHEN:
                if (!this.inNichtsortiert) {
                    this.fehlerPuffer.add(
                            ErrorCode.FEHLENDER_NICHTSORTIERBEGINN_IN_FELD);
                }
                this.puffer.append(inputChar);
                this.inNichtsortiert = false;
                break;
            case MabCharset.SATZENDEZEICHEN:
                this.fehlerPuffer.add(ErrorCode.SATZENDE_IN_FELD);
                // Fuege ein Feldendezeichen hinzu, lege das gelesene
                // Satzendezeichen zurueck.
                // Feld wird beim Lesen des eingefuegten Felendezeichens
                // gespeichert.
                this.toInsert.add(MabCharset.FELDENDEZEICHEN);
                this.toInsert.add(inputChar);
                break;
            case MabCharset.STICHWORTBEGINNZEICHEN:
                if (this.inStichwort) {
                    this.fehlerPuffer.add(
                            ErrorCode.WIEDERHOLTER_STICHWORTBEGINN_IN_FELD);
                }
                this.inStichwort = true;
                this.puffer.append(inputChar);
                break;
            case MabCharset.STICHWORTENDEZEICHEN:
                if (!this.inStichwort) {
                    this.fehlerPuffer.add(
                            ErrorCode.FEHLENDER_STICHWORTBEGINN_IN_FELD);
                }
                this.inStichwort = false;
                this.puffer.append(inputChar);
                break;
            case MabCharset.TEILFELDTRENNZEICHEN:
                this.puffer.append(inputChar);
                break;
            case MabCharset.UNTERFELDBEGINNZEICHEN:
                if (this.puffer.length() > 0) {
                    // ignoriere Feld
                    // setzt auch this.zustand = IN_Satz und resetFeld()
                    this.ignoriereFeld(this.sFeldnummer
                        + Character.toString(this.feldindikator)
                        + this.puffer.toString(),
                        ErrorCode.FELDINHALT_UND_UNTERFELD_VORHANDEN);
                    break;
                }
                // fügen Feld jetzt ein, da wir die feldID brauchen.
                // Feldendezeichen wird eh in Zustand IN_UNTERFELD gelesen,
                // Feld wird also _nicht_ zweimal eingetragen.
                // ueberpruefe ob Feldnummer und Indikator Mabkonform sind.
                if (!MabFeldDefinition.isMab2Field(this.iFeldnummer,
                        this.feldindikator, false)) {
                    this.fehlerPuffer.add(
                            ErrorCode.FELDNUMMER_INDIKATOR_UNZULAESSIG);
                }

                this.contentHandler.startFeld(this.sFeldnummer,
                            this.feldindikator, null);

                // übertrage Fehler in die Datenbank, leere Fehlerpuffer
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator, this.fehlerPuffer);
                this.fehlerPuffer.clear(); // TODO more?
                this.zustand = Zustand.NACH_UNTERFELDSTART;
                break;
            default:
                // TODO verify check
                if (!MabCharsetChecker.checkUTF8bytes(inputChar)) {
                    this.fehlerPuffer.add(ErrorCode.ZEICHENSATZ_FEHLER);
                }
                this.puffer.append(inputChar);
                break;
        } // switch(inputChar)
    }

    protected void parseNachUnterfeldStart(final char inputChar)
            throws IOException {
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator,
                        ErrorCode.FELDENDE_STATT_UFKENNZEICHEN);
                this.resetFeld();
                this.zustand = Zustand.IN_SATZ;
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator,
                        ErrorCode.NICHTSORTIERBEGINN_STATT_UFKENNZEICHEN);
                this.ignoriereUnterfeld(Character.toString(inputChar),
                        ErrorCode.NICHTSORTIERBEGINN_STATT_UFKENNZEICHEN);
                // setzt auch neuen Zustand und resetet
                break;
            case MabCharset.NICHTSORTIERENDEZEICHEN:
                this.ignoriereUnterfeld(Character.toString(inputChar),
                        ErrorCode.NICHTSORTIERENDE_STATT_UFKENNZEICHEN);
                // setzt auch neuen Zustand und resetet
                break;
            case MabCharset.SATZENDEZEICHEN:
                this.errorHandler.insertSatzfehler(ErrorCode.SATZENDE_STATT_UFKENNZEICHEN);
                this.resetUnterfeld();
                this.resetFeld();
                this.beendeSatz(); // setzt auch Zustand IN_DOKUMENT
                break;
            case MabCharset.STICHWORTBEGINNZEICHEN:
                this.ignoriereUnterfeld(Character.toString(inputChar),
                        ErrorCode.STICHWORTBEGINN_STATT_UFKENNZEICHEN);
                break;
            case MabCharset.STICHWORTENDEZEICHEN:
                this.ignoriereUnterfeld(Character.toString(inputChar),
                        ErrorCode.STICHWORTENDE_STATT_UFKENNZEICHEN);
                break;
            case MabCharset.TEILFELDTRENNZEICHEN:
                this.ignoriereUnterfeld(Character.toString(inputChar),
                        ErrorCode.TEILFELDTRENNUNG_STATT_UFKENNZEICHEN);
                break;
            case MabCharset.UNTERFELDBEGINNZEICHEN:
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator,
                        ErrorCode.UNTERFELDBEGINN_STATT_UFKENNZEICHEN);
                break;
            default:
                this.unterfeldkennzeichen = inputChar;
                if (!(Character.toString(this.unterfeldkennzeichen).matches(
                        "[a-zA-Z0-9 ]"))) {
                    this.ignoriereUnterfeld(Character.toString(inputChar),
                            ErrorCode.UFKENNZEICHEN_FORMATFEHLER);
                    // setzt auch neuen Zustand
                    break;
                }
                this.zustand = Zustand.IN_UNTERFELD;
                break;
        } // switch(inputChar)
    }

    protected void parseInUnterfeld(final char inputChar) {
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                this.beendeUnterfeld();
                this.resetFeld();
                this.zustand = Zustand.IN_SATZ;
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
                if (this.inNichtsortiert) {
                    this.fehlerPuffer.add(
                            ErrorCode.WIEDERHOLTER_NICHTSORTIERBEGINN_IN_UFELD);
                }
                this.inNichtsortiert = true;
                this.puffer.append(inputChar);
                break;
            case MabCharset.NICHTSORTIERENDEZEICHEN:
                if (!this.inNichtsortiert) {
                    this.fehlerPuffer.add(
                            ErrorCode.FEHLENDER_NICHTSORTIERBEGINN_IN_UFELD);
                }
                this.inNichtsortiert = false;
                this.puffer.append(inputChar);
                break;
            case MabCharset.SATZENDEZEICHEN:
                this.fehlerPuffer.add(ErrorCode.SATZENDE_IN_UFELD);
                // Feldendezeichen einfuegen, Satzendezeichen zurueck legen.
                this.toInsert.add(MabCharset.FELDENDEZEICHEN);
                this.toInsert.add(inputChar);
                break;
            case MabCharset.STICHWORTBEGINNZEICHEN:
                if (this.inStichwort) {
                    this.fehlerPuffer.add(
                            ErrorCode.WIEDERHOLTER_STICHWORTBEGINN_IN_UFELD);
                }
                this.inStichwort = true;
                this.puffer.append(inputChar);
                break;
            case MabCharset.STICHWORTENDEZEICHEN:
                if (!this.inStichwort) {
                    this.fehlerPuffer.add(
                            ErrorCode.FEHLENDER_STICHWORTBEGINN_IN_UFELD);
                }
                this.inStichwort = false;
                this.puffer.append(inputChar);
                break;
            case MabCharset.TEILFELDTRENNZEICHEN:
                this.puffer.append(inputChar);
                break;
            case MabCharset.UNTERFELDBEGINNZEICHEN:
                this.beendeUnterfeld();
                this.zustand = Zustand.NACH_UNTERFELDSTART;
                break;
            default:
                // TODO verify check
                if (!MabCharsetChecker.checkUTF8bytes(inputChar)) {
                    this.fehlerPuffer.add(ErrorCode.ZEICHENSATZ_FEHLER);
                }
                this.puffer.append(inputChar);
                break;
        } // switch(inputChar)
    }

    /**
     * Parses characters of ignored dataset until its end is reached.
     *
     * Sätze werden ignoriert, wenn es Fehler in der Satzkennung gibt.
     * - formale Satzlängeangabe (keine Prüfung ob der Wert stimmt)
     * - Satzstatus
     * - Indikatorlänge
     * - Data start address falsch
     *
     * Wenn diese Funktion aufgerufen wird, wurde gerade die Satzkennung gelesen.
     * Das heißt die ersten Zeichen sind normalerweise bereits das 001 Feld.
     *
     * @param inputChar Current character
     * @throws IOException if error reading data occurs
     */
    protected void parseIgnoredDataset(final char inputChar)
            throws IOException {
        this.datenIgnoriert = true;
        this.printSatzInfos = true;
        // bei ignorierten Sätzen wird jedes Zeichen in den Puffer aufgenommen,
        // um den ignorierten Satz vollständig in die LOG-Datei schreiben zu
        // können.
        this.puffer.append(inputChar);
        switch (inputChar) {
            case MabCharset.FELDENDEZEICHEN:
                checkForField001(inputChar);
                break;
            case MabCharset.SATZENDEZEICHEN:
                // Satzendezeichen, gebe ignorierten Satz aus, Verlasse Zustand
                // Ingoriere_satz, beginne mit naechstem Satz.
                this.errorHandler.rejectDataset(this.ignoredDatasetError,
                        this.ignoredDatasetId, this.puffer.toString(), false);
                this.ignoredDatasetError = null;
                this.ignoredDatasetId = null;
                this.resetFeld(); // leert auch den Puffer
                ++this.saetzeIgnoriert;
                this.printSatzInfos = false;
                // fehlerpuffer leeren
                this.fehlerPuffer.clear();
                this.zustand = Zustand.IN_DOKUMENT;
                break;
            case MabCharset.NICHTSORTIERBEGINNZEICHEN:
            case MabCharset.NICHTSORTIERENDEZEICHEN:
            case MabCharset.STICHWORTBEGINNZEICHEN:
            case MabCharset.STICHWORTENDEZEICHEN:
            case MabCharset.TEILFELDTRENNZEICHEN:
            case MabCharset.UNTERFELDBEGINNZEICHEN:
            default:
                if (checkFirstField) {
                    checkForField001(inputChar);
                    checkFirstField = false;
                }
                // nichts zu tun, warten auf Satzendezeichen
                break;
        } // switch(inputChar)
    }

    protected void checkForField001(final char inputChar) throws IOException {
        int fnr2 = this.nextToken();
        int fnr3 = this.nextToken();
        if (fnr2 == -1 || fnr3 == -1) {
            // Dateiende erreicht, Fehlermeldung über unvollständigen
            // Satz wird unten gegeben.
            return;
        }
        // Pruefe auf Satzendezeichen. Lege gelesene Zeichen ggf.
        // zurück.
        if (((char) fnr2) == MabCharset.SATZENDEZEICHEN) {
            this.toInsert.add((char) fnr2);
            this.toInsert.add((char) fnr3);
            return;
        }
        if ((char) fnr3 == MabCharset.SATZENDEZEICHEN) {
            // übernehme alle Zeichen, die nicht wieder zurückgelegt
            // werden zur Ausgabe.
            this.puffer.append((char) fnr2);
            this.toInsert.add((char) fnr3);
            return;
        }

        // übernehme alle Zeichen, die nicht wieder zurückgelegt werden
        // zur Ausgabe.
        this.puffer.append((char)fnr2);
        this.puffer.append((char)fnr3);

        // pruefe ob Feld 001
        if (inputChar == '0' && ((char) fnr2) == '0'
                && ((char) fnr3) == '1') {
            // Pruefe indikator.
            int iIndikator = this.nextToken();
            if (iIndikator == -1) {
                // Dateiende erreicht, Fehlermeldung über
                // unvollständigen Satz wird unten gegeben.
                return;
            }
            char cIndikator = (char) iIndikator;
            // pruefe auf Satzendezeichen, lege ggf. gelesenes Zeichen
            // zurueck
            if (cIndikator == MabCharset.SATZENDEZEICHEN) {
                this.toInsert.add(cIndikator);
                return;
            }
            this.puffer.append(cIndikator);
            // pruefe auf indikator = blank
            if (cIndikator == ' ') {
                // extrahiere Feldinhalt
                String feldinhalt = "";
                int input;
                do {
                    input = this.nextToken();
                    feldinhalt += Character.toString((char) input);
                }
                while (input != -1
                        && input != MabCharset.FELDENDEZEICHEN
                        && input != MabCharset.SATZENDEZEICHEN);
                if (input == MabCharset.SATZENDEZEICHEN) {
                    // kein Feldendezeiche => Feld defekt, legen
                    // Satzendezeichen zurueck.
                    this.puffer.append(feldinhalt);
                    this.toInsert.add((char) input);
                    return;
                }
                else if (input == -1) {
                    this.puffer.append(feldinhalt);
                    return;
                }
                if (feldinhalt.length() > 0) {
                    this.ignoredDatasetId = feldinhalt.substring(0,
                            feldinhalt.length() - 1);
                    this.puffer.append(feldinhalt);
                }
            }
        }
    }

    protected void finishParsing() throws ParseException {
        // EOF bekommen, Parsen abschliessen
        switch (this.zustand) {
            case IN_DOKUMENT:
                break;
            case IN_FELD:
                this.fehlerPuffer.add(ErrorCode.EOF_IN_FELD);
                commonParseCode();
                // Feld eintragen
                this.contentHandler.startFeld(this.sFeldnummer,
                        this.feldindikator, this.puffer.toString());

                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator, this.fehlerPuffer);
                // zuruecksetzen
                this.resetFeld();
                this.beendeSatz();
                this.datenIgnoriert = true;
                break;
            case IN_SATZ:
                this.beendeSatz();
                this.errorHandler.insertLieferungsfehler(
                        ErrorCode.EOF_IN_SATZ);
                this.datenIgnoriert = true;
                break;
            case IGNORIERE_SATZ:
                this.datenIgnoriert = true;
                this.errorHandler.rejectDataset(this.ignoredDatasetError,
                        this.ignoredDatasetId, this.puffer.toString(), true);
                this.ignoredDatasetError = null;
                this.ignoredDatasetId = null;
                ++this.saetzeIgnoriert;
                break;
            case NACH_UNTERFELDSTART:
                this.errorHandler.insertFeldfehler(sFeldnummer + feldindikator,
                        ErrorCode.EOF_STATT_UFKENNZEICHEN);
                this.beendeSatz();
                this.datenIgnoriert = true;
                break;
            case UNDEFINIERT:
            default:
                throw new ParseException("Parsen abgeschlossen, ohne Datei zu "
                        + "öffnen. Zustand des Parsers ist undefiniert. Bitte "
                        + "den Entwickler kontaktieren.", this.position);
        } // switch(this.zustand)
        if (this.datenIgnoriert) {
            this.errorHandler.insertLieferungsfehler(
                    ErrorCode.LIEFERUNG_UNVOLLSTAENDIG);
        }
        // this.contentHandler.flush(); // TODO was in old interface
        this.contentHandler.endParsing();
    }

    protected void commonParseCode() {
        if (this.inNichtsortiert) {
            this.fehlerPuffer.add(ErrorCode.FEHLENDES_NICHTSORTIERENDE_IN_FELD);
        }
        if (this.inStichwort) {
            this.fehlerPuffer.add(ErrorCode.FEHLENDES_STICHWORTENDE_IN_FELD);
        }
        if (!MabFeldDefinition.isMab2Field(this.iFeldnummer, this.feldindikator,
                true)) {
            // ueberpruefe ob es sich um einen Unterfeldfehler handelt
            if (!MabFeldDefinition.isMab2Field(this.iFeldnummer,
                    this.feldindikator, false)) {
                this.fehlerPuffer.add(
                        ErrorCode.FELDNUMMER_INDIKATOR_UNZULAESSIG);
            }
            else {
                this.fehlerPuffer.add(ErrorCode.FEHLENDES_UNTERFELD);
            }
        }
    }

    /**
     * Hilfsmethode, zum auslesen von Zeichen. Wird gebraucht, damit Zeichen
     * zurückgelegt und eingefügt werden können.
     * @return das nächste zu lesende Zeichen (entweder aus {@link #toInsert}
     *         oder aus {@link #in}).
     * @throws IOException
     */
    private int nextToken() throws IOException {
        int newChar;

        if (this.toInsert.size() > 0) {
            newChar = (int) this.toInsert.poll();
        }
        else {
            ++this.position;
            newChar = this.in.read();
        }

        // ignoriere Zeilenumbrueche
        if (newChar == LINEFEED || newChar == CARRIAGERETURN) {
            newChar = this.nextToken();
        }

        return newChar;
    }

    /**
     * Hilfsmethode, liest bis zum nächsten Feldende, schreibt das Feld in die
     * Reject-Datei, ruft {@link #resetFeld()} auf und setzt {@code #zustand}
     * auf {@code Zustand#IN_DOKUMENT}.
     * @throws IOException
     */
    private void ignoriereFeld(final String fieldData, final ErrorCode error)
            throws IOException {
        String fetchedFieldData = fieldData;
        this.datenIgnoriert = true;
        this.printSatzInfos = true;
        int input;
        char inputChar;
        do {
            input = this.nextToken();
            inputChar = (char) input;
            fetchedFieldData += Character.toString(inputChar);
        }
        while (input != -1 && inputChar != MabCharset.FELDENDEZEICHEN
                && inputChar != MabCharset.SATZENDEZEICHEN);

        this.errorHandler.rejectField(error, fetchedFieldData);

        if (input == -1) {
            // TODO handle as package error
            this.errorHandler.addRejectError(ErrorCode.EOF_IN_FELD);
            return;
        }
        else if (inputChar == MabCharset.SATZENDEZEICHEN) {
            // TODO handle as package error
            this.errorHandler.addRejectError(ErrorCode.SATZENDE_IN_FELD);
            this.resetFeld();
            this.beendeSatz(); // setzt Zustand IN_DOKUMENT
            return;
        }
        // Haben Feldendezeichen eines kaputten Feldes gelesen:
        this.resetFeld();
        this.zustand = Zustand.IN_SATZ;
    }

    /**
     * Setzt alle Variablen eines Feldes zurück.
     *
     * Sollte nach dem einlesen eines Feldes aufgerufen werden.
     */
    private void resetFeld() {
        this.feldindikator = '#';
        this.iFeldnummer = -1;
        this.sFeldnummer = "";
        this.inNichtsortiert = false;
        this.inStichwort = false;
        this.unterfeldkennzeichen = '#';
        this.puffer.setLength(0);
        this.fehlerPuffer.clear();
    }

    /**
     * Liest alle Daten bis zum Ende des Unterfeldes, schreibt die Inhalte in
     * die Reject-Datei und setzt das Unterfeld zurück. Setzt den Zustand auf
     * {@code Zustand#NACH_UNTERFELDSTART} oder {@code Zustand#IN_SATZ} oder
     * ruft {@link #beendeSatz()} auf.
     * @param fetchedSubfieldData alle bis dahin gelesenen Inhalte des Unterfeldes
     * @throws IOException
     */
    private void ignoriereUnterfeld(final String fetchedSubfieldData,
            final ErrorCode error) throws IOException {
        String subfieldData = fetchedSubfieldData;
        this.datenIgnoriert = true;
        this.printSatzInfos = true;
        int input;
        char inputChar;
        do {
            input = this.nextToken();
            inputChar = (char) input;
            subfieldData += Character.toString(inputChar);
        }
        while (input != -1 && inputChar != MabCharset.FELDENDEZEICHEN
                && inputChar != MabCharset.UNTERFELDBEGINNZEICHEN
                && inputChar != MabCharset.SATZENDEZEICHEN);

        this.errorHandler.rejectSubfield(error, subfieldData);

        if (input == -1) {
            // TODO handle as package error
            this.errorHandler.addRejectError(ErrorCode.EOF_IN_UFELD);
            // EOF wird von alleine gefunden
        }
        else if (inputChar == MabCharset.FELDENDEZEICHEN) {
            this.resetUnterfeld();
            this.resetFeld();
            this.zustand = Zustand.IN_SATZ;
        }
        else if (inputChar == MabCharset.UNTERFELDBEGINNZEICHEN) {
            this.resetUnterfeld();
            this.zustand = Zustand.NACH_UNTERFELDSTART;
        }
        else {
            // SATZENDEZEICHEN
            // TODO handle as dataset error
            this.errorHandler.addRejectError(ErrorCode.SATZENDE_IN_UFELD);
            this.resetUnterfeld();
            this.resetFeld();
            this.beendeSatz(); // setzt auch zustand = IN_DOKUMENT
        }
    }

    /**
     * Setzt alle Variablen eines Unterfeldes zurück. Sollte immer aufgerufen
     * werden, wenn ein Unterfeld vollständig eingelesen wurde.
     */
    private void resetUnterfeld() {
        this.inNichtsortiert = false;
        this.inStichwort = false;
        this.unterfeldkennzeichen = '#';
        this.puffer.setLength(0); // TODO = new StringBuilder();
        this.fehlerPuffer.clear();
    }

    /**
     * Schließt ein Unterfeld ab und Speichert es in der Datenbank.
     *
     * Ruft {@code #resetUnterfeld()} auf.
     */
    private void beendeUnterfeld() {
        if (!MabFeldDefinition.isMabSubfield(this.iFeldnummer,
                this.feldindikator, this.unterfeldkennzeichen)) {
            this.fehlerPuffer.add(ErrorCode.UNZULAESSIGES_UNTERFELD);
        }
        this.contentHandler.startUnterfeld(this.unterfeldkennzeichen,
                puffer.toString());
        this.errorHandler.insertUnterfeldfehler(this.fehlerPuffer);
        this.resetUnterfeld();
    }

    /**
     * Schließt einen Satz ab, gibt Satzinfos aus, falls {@code printSatzInfos}
     * gesetzt ist, ruft {@code #resetFeld()} auf und setzt {@code #zustand} auf
     * {@code Zustand#IN_DOKUMENT}.
     */
    private void beendeSatz() {
        ++this.saetzeGefunden;
        if (this.printSatzInfos) {
            this.errorHandler.endReject();
            this.printSatzInfos = false;
        }
        this.contentHandler.endSatz();
        this.resetFeld();
        this.zustand = Zustand.IN_DOKUMENT;
    }

    /**
     *
     * @return Number of datasets parsed.
     */
    @Override
    public int getDatasetsProcessed() {
        return this.saetzeGefunden;
    }

    @Override
    public int getDatasetsIgnored() {
        return this.saetzeIgnoriert;
    }

    @Override
    public void setSource(InputStream stream) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public ContentHandler getContentHandler() {
        return contentHandler;
    }

}