/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.extension;

import de.kobv.mable.mab.MabFeldDefinition;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * Klasse für Fehlertypen.
 *
 * Jeder Instanz von ErrorType repräsentiert einen Mable+ Fehler. Diese Klasse ist notwendig, da Instanzen der Klasse
 * DatasetError konkrete, gefundene Fehler repräsentieren und mit Sätzen verknüpft sind.
 *
 * Jede Instanz wird mit allen notwendigen Informationen wie Fehlercode und Nachricht erzeugt.
 *
 * DEV FÜr jeden Fehlertyp sollte nur eine Instanz dieser Klasser erzeugt werden. Es werden also FactoryMethods
 * verwendet.
 *
 * TODO Support messageKey?
 * TODO generate error codes if necessary
 * TODO Verhalten, wenn newInstance mit identischem identifier aber unterschiedlicher Message und Code aufgerufen wird
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 */
abstract public class ErrorType implements Comparable<ErrorType> {

    /**
     * Logger for this class.
     */
    private final static Logger LOG = Logger.getLogger(ErrorType.class);

    /**
     * Unique identifier for error type.
     */
//     private int errorHash;

    /**
     * Error code for reports.
     */
    private int errorCode;

    private boolean errorCodeRegistered = false;

    /**
     * Message for error type.
     */
    private String message;

    private static Properties messages;


    /**
     * TODO refactor! Improve design if possible.
     */
    private final static ErrorCodeManager errorCodeFactory = new ErrorCodeManager();

    public ErrorType() {
        this(null, 0);
    }

    public ErrorType(final String message) {
        this(message, 0);
    }

    public ErrorType(final String message, final int code) {
        this.errorCode = code;
        this.message = message;
    }

    /**
     *
     * @param identifier
     * @param message
     * @param code
     * @return
     */
    /** public static ErrorType newInstance(final int identifier, final String message, final int code)
            throws ErrorCodeAlreadyExistsException {
        Map<Integer, ErrorType> types = getErrorTypes();

        ErrorType errorType = types.get(identifier);

        if (errorType == null) {
            int errorCode = code;

            errorType = new ErrorType(identifier, message, errorCode);
            types.put(identifier, errorType);
        }

        return errorType;
    } */

    /**
     * Returns unique hashcode for error type.
     * @return  Hash for ErrorType class
     */
    public abstract int getHash();

    public int getCodeHash() {
        return getHash();
    }

    public int getCode() {
        if (!errorCodeRegistered) {

            if (errorCode != 0) {
                try {
                    errorCodeFactory.registerCode(getCodeHash(), errorCode);
                }
                catch (ErrorCodeAlreadyExistsException e) {
                    errorCode = errorCodeFactory.generateCode(getCodeHash());
                    // TODO log warning
                }
            }
            else {
                errorCode = errorCodeFactory.generateCode(getCodeHash());
            }

            errorCodeRegistered = true;
        }
        return errorCode;
    }

    public String getMessage() {
        String msg = message;

        Properties messages = getMessages();

        if (message != null && messages.containsKey(message)) {
            msg = messages.getProperty(message);
        }

        return msg;
    }

    /**
     * Returns hash code for use by collections.
     * @return int
     */
    @Override
    public int hashCode() {
        return getHash();
    }

    @Override
    public int compareTo(final ErrorType o) {
        return Integer.compare(getCode(), o.getCode());
    }

    @Override
    public boolean equals(Object obj) {
        return getClass() == obj.getClass() && hashCode() == obj.hashCode();
    }

    /**
     * Loads error messages associated with error codes.
     * @return Properties instance containing error messages
     *
     * TODO improvements? configurable?
     */
    public static Properties getMessages() {
        if (messages == null) {
            messages = new Properties();
            try {
                InputStream in = ErrorType.class.getClassLoader().getResourceAsStream("de/kobv/mable/messages.xml");
                messages.loadFromXML(in);
            }
            catch (IOException ioe) {
                LOG.error("messages.xml not found");
                // TODO do something? fatal error, should not happen
            }
        }
        return messages;
    }

    public static String getMessage(String key) {
        Properties messages = getMessages();

        if (messages.containsKey(key)) {
            return messages.getProperty(key);
        }

        return key;

    }

    protected String getFeldname(String kategorie) {
        try {
            MabFeldDefinition feldDef = MabFeldDefinition.valueOf("MAB" + kategorie);
            if (feldDef != null) {
                return feldDef.getName();
            }
        }
        catch (IllegalArgumentException iae) {
            // Kategorie (z.B. "LOW" existiert nicht in MabFeldDefinition)
        }
        return null;
    }

}
