/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab.extension;

import de.kobv.mable.ErrorCode;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.*;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

/**
 * Counts the occurences of different error types.
 *
 * Logs dataset identifiers for errors in order to provide a mechanism for finding the datasets.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO refactor to work with classes rather than codes
 * TODO remove?
 * TODO refactor
 */
public class ErrorStatistics implements IErrorStatistics {

    private final static Logger LOG = Logger.getLogger(ErrorStatistics.class);

    private Map<ErrorType, Integer> fehler = new HashMap<ErrorType, Integer>();

    private ErrorLogger errorLog;

    private String current001;

    @Override
    public void addError(DatasetError error) {
        ErrorType errorType = error.getErrorType();

        if (errorType == null) {
            System.out.println(this + " error without type : " + error);
            throw new NullPointerException("error without error type");
        }

        int count = 0;

        if (fehler.containsKey(errorType)) {
            count = fehler.get(errorType);
        }
        fehler.put(errorType, count + 1);

        errorLog.logError(error);
    }

    protected void addError(final ErrorType errorType) {
        int count = 0;

        if (fehler.containsKey(errorType)) {
            count = fehler.get(errorType);
        }
        fehler.put(errorType, count + 1);
    }

    public void addFieldError(final String fieldType, final ErrorCode errorCode) {
        ErrorType errorType = new FieldErrorType(fieldType, errorCode);
        addError(errorType);
        errorLog.logError(getError(errorType));
    }

    public void addFormatError(final ErrorCode errorCode) {
        ErrorType errorType = new FormatErrorType(errorCode);
        addError(errorType);
        errorLog.logError(getError(errorType));
    }

    public void addPackageError(final ErrorCode errorCode) {
        ErrorType errorType = new PackageErrorType(errorCode);
        addError(errorType);
        errorLog.logError(getError(errorType));
    }

    public void addSubfieldError(final ErrorCode errorCode) {
        ErrorType errorType = new SubfieldErrorType(errorCode);
        addError(errorType);
        errorLog.logError(getError(errorType));
    }

    /**
     * Liefert alle Fehler mit einer bestimmten Fehlerklasse zurück.
     * @param errorTypeClass Type of errors that should be returned
     * @return Mapping von Fehlertypen und Anzahl der gefundenen Fehler
     */
    public Map<ErrorType, Integer> getErrors(Class errorTypeClass) {
        Map<ErrorType, Integer> result = new TreeMap<ErrorType, Integer>();

        for (Map.Entry<ErrorType, Integer> entry : fehler.entrySet()) {
            if (errorTypeClass.isInstance(entry.getKey())) {
                result.put(entry.getKey(), 1); //
            }
        }

        return result;
    }

    /**
     * Liefert alle Lieferungsfehler zurück.
     * @return Mapping mit Lieferungsfehlern und der Anzahl der gefundenen Fehler
     */
    public Map<ErrorType, Integer> getPackageErrors() {
        return getErrors(PackageErrorType.class);
    }

    /**
     * Liefert alle Formatfehler zurück.
     * @return Mapping mit Formatfehler und Anzahl der gefundenen Fehler
     */
    public Map<ErrorType, Integer> getFormatErrors() {
        return getErrors(FormatErrorType.class);
    }

    /**
     * Liefert alle Feldfehler zurück.
     * @return Mapping der Feldfehler und der Anzahl der gefundenen Fehler
     */
    public Map<FieldErrorType, Integer> getFieldErrors() {
        Map<FieldErrorType, Integer> result = new TreeMap<FieldErrorType, Integer>(new FieldErrorTypeComparator());

        for (Map.Entry<ErrorType, Integer> entry : fehler.entrySet()) {
            if (entry.getKey() instanceof FieldErrorType) {
                result.put(( FieldErrorType )entry.getKey(), entry.getValue());
            }
        }
        return result;
    }

    /**
     * Liefert alle Unterfeldfehler zurück.
     * @return Mapping der Unterfeldfehler und der Anzahl der gefundenen Fehler
     */
    public Map<ErrorType, Integer> getSubfieldErrors() {
        return getErrors(SubfieldErrorType.class);
    }

    /**
     * Returns a Map with keys sorted by error codes.
     * @return
     *
     * TODO kompiliert, aber funktioniert noch nicht.
     */
    public Map<ErrorType, Integer> getDatasetErrors() {
        Map<ErrorType, Integer> result = new TreeMap<ErrorType, Integer> ();

        for (Map.Entry<ErrorType, Integer> entry : fehler.entrySet()) {
            if (!(entry.getKey() instanceof CodeErrorType)) {
                result.put(entry.getKey(), entry.getValue());
            }
        }

        return result;
    }

    /**
     * @param out
     */
    public void writeIndex(PrintWriter out) {
        for (ErrorType errorType : fehler.keySet()) {
            out.println(String.format("#%1$s, %2$d, %3$s", errorType.getCode(), errorType.getHash(), errorType.getMessage()));
        }
    }

    /**
     * Setzt Satz-ID (MAB001) für Satz der gerade verarbeitet wird.
     *
     * Dies ist notwendig, da der Parser nur Fehlercodes meldet und daher nicht direkt festgestellt werden kann um
     * welchen Satz es sich handelt.
     *
     * @param current001
     */
    public void setCurrentDataset001(final String current001) {
        this.current001 = current001;
    }

    @Autowired
    @Required
    public void setErrorLog(ErrorLogger errorLog) {
        this.errorLog = errorLog;
    }

    /**
     * Liefert DatasetError Instanz für Fehlercode.
     *
     * Diese Klasse konvertiert den alten Ansatz mit Fehlercodes in den neuen Ansatz Fehler mit Hilfe von DatasetError
     * Instanzen zu melden.
     *
     * @param errorType Fehlertyp
     * @return DataSetError Fehlerinstanz
     */
    private DatasetError getError(ErrorType errorType) {
        return new DatasetError(errorType, current001);
    }

}
