/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import java.util.Arrays;

/**
 * Parses and verifies MAB Satzkennung.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO create unit test for this class
 */
public final class MabSatzkennung {

    /**
     * Length of MAB2 header.
     */
    public static final int HEADER_LENGTH = 24;

    /**
     * Allowed characters for dataset types.
     *
     * TODO figure out what is faster: array or string
     */
    public static final char[] DATASET_TYPES = {
        'h', 'u', 'y', 'v', 'p', 't', 'k', 'w', 'r', 's', 'x', 'q', 'l', 'e',
        'm', 'c'
    };

    /**
     * Allowed characters for dataset types as String.
     */
    public static final String DATASET_TYPES_STR = "huyvptkwrsxqlemc";

    /**
     * Allowed characters for dataset status.
     */
    public static final char[] DATASET_STATUS = {
        'c', 'd', 'n', 'p', 'u', 'v'
    };

    /**
     * Index for length of dataset.
     */
    public static final int LENGTH_INDEX = 0;

    /**
     * Index of status character.
     */
    public static final int STATUS_INDEX = 5;

    /**
     * Index for version information in header.
     */
    public static final int VERSION_INDEX = 6;

    /**
     * Index for length of indicator.
     */
    public static final int INDICATOR_LENGTH_INDEX = 10;

    /**
     * Index for length of subfield header.
     */
    public static final int SUBFIELD_ID_LENGTH_INDEX = 11;

    /**
     * Index for offset value for data.
     */
    public static final int DATA_OFFSET_INDEX = 12;

    /**
     * Index for reserved fields in header.
     */
    public static final int RESERVED_FIELDS_INDEX = 17;

    /**
     * Index for dataset type field in header.
     */
    public static final int TYPE_INDEX = 23;

    /**
     * Allowed characters for dataset status as String.
     */
    public static final String DATASET_STATUS_STR = "cdnpuv";

    /**
     * Sort arrays if characters for binary search.
     */
    static {
        Arrays.sort(DATASET_TYPES);
        Arrays.sort(DATASET_STATUS);
    }

    /**
     * Entire dataset header as string.
     */
    private String kennung;

    /**
     * Constructs empty dataset header object.
     */
    public MabSatzkennung() {
    }

    public MabSatzkennung(String kennung) {
        setValue(kennung);
    }

    /**
     * Constructs a dataset header from parameters.
     * @param length Length of dataset
     * @param type Type of dataset
     * @param status Status of dataset
     */
    public MabSatzkennung(final int length, final String type,
            final String status) {
        String kennung = MabSatzkennung.constructSatzkennung(length, type,
                status);
        setValue(kennung);
    }

    /**
     * Returns value string for "Satzkennung".
     * @return Satzkennung
     */
    public String getValue() {
        return kennung;
    }
    /**
     * Sets value of dataset header.
     * @param header MAB dataset header
     */
    public void setValue(final String header) {
        this.kennung = header;
    }

    /**
     * Satzkennung überprüfen.
     *
     * @return true - if header has correct length
     * TODO Update javadoc for this method
     */
    public boolean isLengthValid() {
        // CHECKSTYLE_OFF: MagicNumber
        return Character.isDigit(kennung.charAt(LENGTH_INDEX))
            && Character.isDigit(kennung.charAt(LENGTH_INDEX + 1))
            && Character.isDigit(kennung.charAt(LENGTH_INDEX + 2))
            && Character.isDigit(kennung.charAt(LENGTH_INDEX + 3))
            && Character.isDigit(kennung.charAt(LENGTH_INDEX + 4));
        // CHECKSTYLE_ON: MagicNumber
    }

    /**
     * Returns dataset status.
     * @return Status of dataset as char
     */
    public char getStatus() {
        return kennung.charAt(STATUS_INDEX);
    }

    /**
     * Check if status is valid.
     * @return true - if dataset status is valid (>=0)
     */
    public boolean isStatusValid() {
//      if (Arrays.binarySearch(DATASET_STATUS, status) < 0) {
        return DATASET_STATUS_STR.indexOf(kennung.charAt(STATUS_INDEX)) >= 0;
    }

    /**
     * Verify MAB version.
     * @return true - Valid MAB version, false - MAB version not valid.
     */
    public boolean isVersionValid() {
        // CHECKSTYLE_OFF: MagicNumber
        return kennung.charAt(VERSION_INDEX) == 'M'
            && kennung.charAt(VERSION_INDEX + 1) == '2'
            && kennung.charAt(VERSION_INDEX + 2) == '.'
            && kennung.charAt(VERSION_INDEX + 3) == '0';
        // CHECKSTYLE_ON: MagicNumber
    }

    /**
     * Check if indicator lenght is valid.
     * @return true - Indicator length is valid, false - it is not.
     */
    public boolean isIndicatorLengthValid() {
        // Indikatorlaenge
        return kennung.charAt(INDICATOR_LENGTH_INDEX) == '1';
    }

    /**
     * Check if subfield id length is valid.
     * @return true - if the value is "2"
     */
    public boolean isTeilfeldkennungslaengeValid() {
        //Teilfeldkennungslaenge
        return kennung.charAt(SUBFIELD_ID_LENGTH_INDEX) == '2';
    }

    /**
     * Check if start address for data is valid.
     * @return true - if data offset is "00024".
     */
    public boolean isDataStartAddressValid() {
        // CHECKSTYLE_OFF: MagicNumber
        return kennung.charAt(DATA_OFFSET_INDEX) == '0'
            && kennung.charAt(DATA_OFFSET_INDEX + 1) == '0'
            && kennung.charAt(DATA_OFFSET_INDEX + 2) == '0'
            && kennung.charAt(DATA_OFFSET_INDEX + 3) == '2'
            && kennung.charAt(DATA_OFFSET_INDEX + 4) == '4';
        // CHECKSTYLE_ON: MagicNumber
    }

    /**
     * Check if reserved fields are empty.
     * @return true - if all reserved fields are empty.
     */
    public boolean isReservedFieldsEmpty() {
        // CHECKSTYLE_OFF: MagicNumber
        return kennung.charAt(RESERVED_FIELDS_INDEX) == ' '
            && kennung.charAt(RESERVED_FIELDS_INDEX + 1) == ' '
            && kennung.charAt(RESERVED_FIELDS_INDEX + 2) == ' '
            && kennung.charAt(RESERVED_FIELDS_INDEX + 3) == ' '
            && kennung.charAt(RESERVED_FIELDS_INDEX + 4) == ' '
            && kennung.charAt(RESERVED_FIELDS_INDEX + 5) == ' ';
        // CHECKSTYLE_ON: MagicNumber
    }

    /**
     * Returns type of dataset.
     * @return Char indicating MAB dataset type.
     */
    public char getTyp() {
        return kennung.charAt(TYPE_INDEX);
    }

    /**
     * Checks if dataset type is valid.
     * @return true - Valid dataset type, false - Dataset type not valid.
     */
    public boolean isTypeValid() {
//      if (Arrays.binarySearch(DATASET_TYPES, satztyp) < 0) {
        return DATASET_TYPES_STR.indexOf(kennung.charAt(TYPE_INDEX)) >= 0;
    }

    /**
     * @return always true
     * TODO Review this method
     * TODO implement (not used right now)
     */
    public boolean isValid() {
        return isLengthValid() && isTypeValid() && isStatusValid() && isVersionValid() && isIndicatorLengthValid()
                && isTeilfeldkennungslaengeValid() && isDataStartAddressValid();
    }

    /**
     * Returns a dataset header string for parameters.
     *
     * @param length Length of dataset
     * @param type Type of MAB2 dataset
     * @param status Status of MAB2 dataset
     */
    public static String constructSatzkennung(final int length,
            final String type, final String status) {
        return String.format("%1$05d%2$cM2.01200024      %3$c",
                length, status.charAt(0), type.charAt(0));
    }

}
