/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.mab;

import de.ddb.charset.MabCharset;
import java.util.LinkedList;
import java.util.List;

/**
 * Repräsentiert ein MAB-Feld.
 *
 * Achtung: da in der Datenbank Felder enthalten sein können, die
 * sowohl Inhalte, wie auch Unterfelder enthalten können, ist dies
 * hier auch Möglich, wenn auch nicht empfehlenswert.
 *
 * Es ist nicht möglich die Kategorie bzw. Feldnummer und den Indikator eines Feldes nachträglich zu verändern. Felder
 * werden können von einem MabSatz über die Bezeichnung (Kategorie + Indikator) abgefragt werden. Wenn sich diese
 * Attribute nachträglich ändern liesen, müssten bei einer Änderung die Datenstrukturen innerhalb des MabSatz
 * aktualisiert werden. Das Feld müsste im Hash unter einem anderen Schlüssel gespeichert werden.
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO review possible problems if 'unterfelder' is modified and iterate by external class
 * TODO Sollte MabFeld das Interface "List" implementieren (bzw. eine entsprechende Klasse erweitern)?
 */
public final class MabFeld implements IMabFeld {

    /**
     * Ein Char-Array, dass alle möglichen Indikator-Zeichen (a-z, A-Z, 0-9 und
     * Space) enthält.
     * azAZ09s
     */
    public static final char[] ALL = {
        // a-z
        'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
        'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
        // A-Z
        'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
        'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
        // 0-9
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        // space, blank, spartium
        ' '
    };

    /**
     * MAB2 Kategorie des Feldes.
     */
    private String category;

    /**
     * Indikator des Feldes.
     */
    private char indikator;

    /**
     * Liste der Unterfelder.
     */
    private List<IMabUnterfeld> unterfelder;

    /**
     * Inhalt des MAB Feldes.
     */
    private String inhalt;

    /**
     * Creates empty fields with number and indicator.
     * @param number MAB field number
     * @param indicator MAB field indicator
     */
    public MabFeld(final int number, final char indicator) {
        this.category = MabFeldDefinition.getFeldnummerString(number);
        this.indikator = indicator;
        setInhalt(null);
        setUnterfelder(null);
    }

    /**
     * Creates MAB field.
     * @param number MAB field number
     * @param indicator MAB field indicator
     * @param content Content of field
     */
    public MabFeld(final int number, final char indicator,
            final String content) {
        this.category = MabFeldDefinition.getFeldnummerString(number);
        this.indikator = indicator;
        setInhalt(content);
        setUnterfelder(null);
    }

    /**
     * Creates empty MAB field with subfields.
     * @param number Number of field
     * @param indicator Indicator for field
     * @param subfields List of subfields
     */
    public MabFeld(final int number, final char indicator,
            final List<IMabUnterfeld> subfields) {
        this.category = MabFeldDefinition.getFeldnummerString(number);
        this.indikator = indicator;
        setInhalt(null);
        setUnterfelder(subfields);
    }

    /**
     * Creates MAB field with content and subfields (not valid in MAB).
     *
     * ACHTUNG: Felder die sowohl Unterfelder, wie auch Inhalte enthalten sind
     * nicht MAB-konform! Konstrutkion ist möglich, da in die Datenbank
     * aufgenommene Daten auch dann rekonstruierbar sein sollen, wenn sie
     * fehlerhaft sind.
     *
     * @param number Number of field
     * @param indicator Indicator for field
     * @param content Content of field
     * @param subfields MAB subfields
     */
    public MabFeld(final int number, final char indicator, final String content,
            final List<IMabUnterfeld> subfields) {
        this.category = MabFeldDefinition.getFeldnummerString(number);
        this.indikator = indicator;
        setInhalt(content);
        setUnterfelder(subfields);
    }

    public MabFeld(final String category, final char indikator) throws MabException {
        if (category == null || category.length() != 3) {
            throw new MabException("Parameter category must have a length of 3.");
        }
        this.category = category;
        this.indikator = indikator;
        setInhalt(null);
        setUnterfelder(null);
    }

    public MabFeld(final String category, final char indikator, String content) throws MabException {
        if (category == null || category.length() != 3) {
            throw new MabException("Parameter category must have a length of 3.");
        }
        this.category = category;
        this.indikator = indikator;
        setInhalt(content);
        setUnterfelder(null);
    }

    /**
     * Adds a subfield to the MAB field.
     * @param unterfeld Subfield
     */
    @Override
    public void addSubfield(final IMabUnterfeld unterfeld) {
        if (this.unterfelder == null) {
            this.unterfelder = new LinkedList<IMabUnterfeld>();
        }
        this.unterfelder.add(unterfeld);
    }

    @Override
    public void removeSubfield(IMabUnterfeld unterfeld) {
        if (this.unterfelder != null) {
            this.unterfelder.remove(unterfeld);
        }
    }

    /**
     * Returns true if field has subfields.
     * @return true - Field has subfields, false - Field has no subfields.
     */
    @Override
    public boolean hatUnterfelder() {
        return (this.unterfelder != null);
    }

    /**
     * Returns true if field has content.
     * @return true - Field has content, false - Field has no content.
     */
    @Override
    public boolean hatInhalt() {
        return (this.inhalt != null && this.inhalt.length() > 0);
    }

    /**
     * Returns list of subfields.
     * @return List of MabUnterfeld objects or null
     *
     * TODO unterfelder kopieren, so daß der Wert von außen nicht verändert werden kann
     */
    @Override
    public List<IMabUnterfeld> getUnterfelder() {
        return this.unterfelder;
    }

    /**
     * Ersetzt alle Unterfelder.
     * @param subfields Liste mit Unterfeldern
     */
    @Override
    public void setUnterfelder(final List<IMabUnterfeld>subfields) {
        if ((subfields != null) && subfields.size() > 0) {
            this.unterfelder = subfields;
        }
        else {
            this.unterfelder = null;
        }
    }

    /**
     * Returns content of field.
     * @return Content as String
     */
    @Override
    public String getInhalt() {
        return this.inhalt;
    }

    /**
     * Sets the content of the MAB field.
     */
    @Override
    public void setInhalt(final String content) {
        this.inhalt = content;
    }

    /**
     * Returns field number.
     * @return MAB field number
     */
    @Override
    public int getFeldnummer() {
        try {
            return Integer.parseInt(category);
        }
        catch (NumberFormatException nfe) {
            return -1;
        }
    }

    /**
     * Returns indicator for field.
     * @return Indicator as Integer
     * TODO Review why int is used.
     */
    @Override
    public int getIndikator() {
        return this.indikator;
    }

    /**
     * Returns full namee of field (number + indicator).
     * @return Full name of field as String
     */
    @Override
    public String getBezeichnung() {
        return this.category + Character.toString(this.indikator);
    }

    /**
     * Converts entire field into String.
     * @return Field as String
     */
    @Override
    public String toString() {
        String feld = this.getBezeichnung();
        if (this.hatInhalt()) {
            feld += this.inhalt;
        }

        if (this.hatUnterfelder()) {
            for (IMabUnterfeld anUnterfelder : this.unterfelder) {
                feld += anUnterfelder.toString();
            }
        }
        feld += Character.toString(MabCharset.FELDENDEZEICHEN);
        return feld;
    }

}
