/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable;

import java.io.IOException;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Enth&auml),lt alle Fehlernummern von MABle und kann diese in die
 * entsprechenden Fehlermeldungen umwandeln.
 *
 * <p>Die Klasses ErrorCodes ist die einzige Referenz, die alle durch MABle
 * gefundenen Fehler in Nummern umwandeln und diesen Nummern in die
 * entsprechenden Fehlernachrichten umwandeln kann.</p>
 *
 * <p>Die Fehler werden durch die Nummern bzw. die Nachrichten beschrieben und
 * jeweils einer Lieferung, einem Satz, einem Feld oder einem Unterfeld
 * zugeordnet.</p>
 *
 * <p>In der Javadoc-Dokumentation sind die Fehlermeldungen angegeben, wie sie
 * von der Methode {@link #getMessage(int)} erzeugt werden.</p>
 *
 * @author Pascal-Nicolas Becker <becker(at)zib.de>
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO rename ErrorCodes into ErrorCode
 */
public enum ErrorCode {

    /* 1000 - 1999 allgemeine Fehler */
    /** Die Lieferung enthielt Fehler, die Datei war nicht MAB-konform. */
    FEHLERHAFTE_LIEFERUNG(1000), // TODO not used
    /** Das Dateiende wurde unerwartet früh erreicht, die Datei wurde
     * wahrscheinlich unvollständig übertragen. */
    UNVOLLSTAENDIGE_DATEI(1001), // TODO not used
    /** Ein oder mehrere Zeichen entsprachen nicht dem zulässigen
     * Zeichensatz. */
    ZEICHENSATZ_FEHLER(1002),
    // Fehlerfolgen
    /** Bei Aufnahme der Lieferung in die Datenbank waren einige Sätze, Felder
     * und/oder Unterfelder unverständlich. Die Datenbank enthält nur die
     * lesbaren Daten und nicht die ganze Lieferung! */
    LIEFERUNG_UNVOLLSTAENDIG(1051),
    // MAB-Zeichen ausserhalb von Saetzen
    /** Ein Feldendezeichen, das keinem Feld und keinem Satz zugeordnet werden
     * kann, wurde gefunden. */
    FELDENDE_OHNE_SATZ(1100),
    /** Ein Nichtsortierbeginnzeichen, das keinem Feld und keinem Satz
     * zugeordnet werden kann, wurde gefunden. */
    NICHTSORTIERBEGINN_OHNE_SATZ(1101),
    /** Ein Nichtsortierendezeichen, das keinem Feld und keinem Satz zugeordnet
     * werden kann, wurde gefunden. */
    NICHTSORTIERENDE_OHNE_SATZ(1102),
    /** Ein Satzendezeichen, das keinem Feld und keinem Satz zugeordnet werden
     * kann, wurde gefunden. */
    SATZENDE_OHNE_SATZ(1103),
    /** Ein Stichwortbeginnzeichen, das keinem Feld und keinem Satz zugeordnet
     * werden kann, wurde gefunden. */
    STICHWORTBEGINN_OHNE_SATZ(1104),
    /** Ein Stichwortendezeichen, das keinem Feld und keinem Satz zugeordnet
     * werden kann, wurde gefunden. */
    STICHWORTENDE_OHNE_SATZ(1105),
    /** Ein Teilfeldtrennzeichen, das keinem Feld und keinem Satz zugeordnet
     * werden kann, wurde gefunden. */
    TEILFELDTRENNUNG_OHNE_SATZ(1106),
    /** Ein Unterfeldbeginnzeichen, das keinem Feld und keinem Satz zugeordnet
     * werden kann, wurde gefunden. */
    UNTERFELDBEGINN_OHNE_SATZ(1107),
    /* 2000 - 2999 Satzfehler */
    /** Das Dateiende wurde innerhalb eines Satzes erreicht. Es fehlt mindestens
     * ein Satzendezeichen. */
    EOF_IN_SATZ(2000),
    /** Das Dateiende wurde innerhalb der Satzkennung erreicht. */
    EOF_IN_SATZKENNUNG(2001),
    // In frueheren Versionen falsch zu georndete Fehlernummern:
    // 2009, 2010, 2011, 2012 und 2013 nicht verwenden!
    // Siehe 2209-2213.
    // Fehler in der Satzkennung
    /** Die Satzlänge in der Satzkennung besteht nicht nur aus Zahlen. */
    SATZLAENGE_UNZULAESSIG(2100),
    /** Der Satzstatus ist unzulässig (ungleich c, d, n, p, u oder v). */
    SATZSTATUS_UNZULAESSIG(2101),
    /** Die Versionsangabe in der Satzkennung lautet nicht 'MAB 2.0'. */
    VERSIONSANGABE_UNZULAESSIG(2102),
    /** Die Indikatorlänge in der Satzkennung ist unzulässig (ungleich 1). */
    INDIKATORLAENGE_UNZULAESSIG(2103),
    /** Die Teilfeldkennungslänge in der Satzkennung ist unzulässig
     * (ungleich 2). */
    TEILFELDKENNUNGSLAENGE_UNZULAESSIG(2104),
    /** Die Datenanfangsadresse ist unzulässig (ungleich '00024'). */
    DATENANFANGSADRESSE_UNZULAESSIG(2105),
    /** Die reservierten Zeichen der Satzkennung waren nicht 5 Leerzeichen. */
    RESERVIERTEZEICHEN_UNZULAESSIG(2106),
    /** Der Satztyp der Satzkennung ist unzulässig
     * (ungleich h, y, u, v, p, t, k, w, r, s, x, q, l, e, m oder c). */
    SATZTYP_UNZULAESSIG(2107),
    // MAB-Zeichen Nach Satzkennung
    /** Es wurde ein Feldendezeichen gefunden, das keinem Feld zugeordnet werden
     * kann. */
    FELDENDE_OHNE_FELD(2150),
    /** Es wurde ein Nichtsortierbeginnzeichen gefunden, das keinem Feld
     * zugeordnet werden kann. */
    NICHTSORTIERBEGINN_OHNE_FELD(2151),
    /** Es wurde ein Nichtsortierendezeichen gefunden, das keinem Feld
     * zugeordnet werden kann. */
    NICHTSORTIERENDE_OHNE_FELD(2152),
    /** Es wurde ein Stichwortbeginnzeichen gefunden, das keinem Feld zugeordnet
     * werden kann. */
    STICHWORTBEGINN_OHNE_FELD(2153),
    /** Es wurde ein Stichwortendezeichen gefunden, das keinem Feld zugeordnet
     * werden kann. */
    STICHWORTENDE_OHNE_FELD(2154),
    /** Es wurde ein Teilfeldtrennungszeichen gefunden, das keinem Feld
     * zugeordnet werden kann. */
    TEILFELDTRENNUNG_OHNE_FELD(2155),
    /** Es wurde ein Unterfeldbeginnzeichen gefunden, das keinem Feld zugeordnet
     * werden kann. */
    UNTERFELDBEGINN_OHNE_FELD(2156),
    // Fehler in Pflichtfeldern oder Feldabhaengigkeiten
    //Früher wurde folgendem Fehler fehlerhaft 2009 als Fehlernummer zugewiesen.
    /** Verfasserwerk ohne Verfasser (MAB 030 Position 11(1, jedoch kein
     * MAB-Feld 100). */
    VERFASSERWERK_OHNE_PERSON(2209),
    /* Früher wurde folgendem Fehler fehlerhaft 2010 als Fehlernummer
     * zugewiesen.
     */
    /** Urheberwerk ohne Körperschaft (MAB 030 Position 11(2, jedoch kein
     * MAB-Feld 200). */
    URHEBERWERK_OHNE_KOERPERSCHAFT(2210),
    /** Das Dateiende wurde innerhalb eines Feldes erreicht. Es fehlen
     * mindestens ein Feldende- und ein Satzendezeichen. */
    EOF_IN_FELD(3000),
    /** Das Dateiende wurde innerhalb der Feldnummer erreicht. */
    EOF_IN_FELDNUMMER(3001),
    /** Das Dateiende wurde an Stelle des Feldindikators erreicht. */
    EOF_IN_FELDINDIKATOR(3002),
    /** Es fehlt ein Feldendezeichen (Satzendezeichen ohne Feldendezeichen). */
    SATZENDE_IN_FELD(3003),
    /** Das Feld ist falsch strukturiert (fehlerhafte Unterfeldbeginnzeichen).*/
    FELDINHALT_UND_UNTERFELD_VORHANDEN(3004),
    // Fehler in Nummer oder Indikator
    /** Die Feldnummer besteht nicht nur aus Zahlen. */
    FELDNUMMER_FORMATFEHLER(3100),
    /** Der Feldindikator ist kein Zeichen aus den Bereichen a-z, A-Z, 0-9 oder
     * Leerzeichen. */
    FELDINDIKATOR_FORMATFEHLER(3101),
    /** Die Kombination aus Feldnummer und Indikator ist im MAB-Standard nicht
     * vorgesehen. */
    FELDNUMMER_INDIKATOR_UNZULAESSIG(3150),
    /** Das Feld hat laut MAB-Standard keine Unterfelder oder das
     * Unterfeldkennzeichen existiert laut MAB-Standard bei diesem Feld nicht.*/
    UNZULAESSIGES_UNTERFELD(3151),
    /** Das Feld ist leer. */
    LEERES_FELD(3152),
    /** Es fehlen Unterfelder, die Daten sind direkt im Feld, statt in
     * Unterfeldern gespeichert. */
    FEHLENDES_UNTERFELD(3153),
    /** Es wurde ein Nichtsortierendezeichen in einem Feld gefunden, ohne
     * passendes Nichtsortierbeginnzeichen. */
    FEHLENDER_NICHTSORTIERBEGINN_IN_FELD(3200),
    /** Es wurde ein Stichwortendezeichen in einem Feld gefunden, ohne
     * passendes Stichwortbeginnzeichen. */
    FEHLENDER_STICHWORTBEGINN_IN_FELD(3201),
    /** Im Feld wurde ein Nichtsortierbereich nicht richtig beendet. Es fehlt
     * ein Nichtsortierendezeichen. */
    FEHLENDES_NICHTSORTIERENDE_IN_FELD(3202),
    /** Im Feld wurde ein Stichwortbereich nicht richtig beendet. Es fehlt ein
     * Stichwortendezeichen. */
    FEHLENDES_STICHWORTENDE_IN_FELD(3203),
    /** Im Feld wurde ein Nichtsortierbeginnzeichen in einem Nichtsortierbereich
     * gefunden. */
    WIEDERHOLTER_NICHTSORTIERBEGINN_IN_FELD(3204),
    /** Im Feld wurde ein Stichwortbeginnzeichen in einem Stichwortbeginnbereich
     * gefunden. */
    WIEDERHOLTER_STICHWORTBEGINN_IN_FELD(3205),
    /* 4000 - 4999 Unterfeldfehler */
    /** Das Dateiende wurde in einem Unterfeld erreicht. */
    EOF_IN_UFELD(4000),
    /** Ein Satzendezeichen wurde in einem Unterfeld gefunden. Es fehlt
     * mindestens ein Feldendezeichen. */
    SATZENDE_IN_UFELD(4001),
    /** Das Dateiende wurde direkt nach einem Unterfeldbeginnzeichen erreicht.*/
    EOF_STATT_UFKENNZEICHEN(4002),
    /** Das Feld endete nach einem Unterfeldbeginnzeichen. */
    FELDENDE_STATT_UFKENNZEICHEN(4050),
    /** Es wurde ein Nichtsortierbeginnzeichen gefunden, an Stelle eines
     * Unterfeldkennzeichens. */
    NICHTSORTIERBEGINN_STATT_UFKENNZEICHEN(4051),
    /** Es wurde ein Nichtsortierendezeichen gefunden, an Stelle eines
     * Unterfeldkennzeichens. */
    NICHTSORTIERENDE_STATT_UFKENNZEICHEN(4052),
    /** Es wurde ein Satzendezeichen gefunden, an Stelle eines
     * Unterfeldbeginnzeichens. */
    SATZENDE_STATT_UFKENNZEICHEN(4053),
    /** Es wurde ein Stichwortbeginzeichen an Stelle eines Unterfeldkennzeichens
     * gefunden. */
    STICHWORTBEGINN_STATT_UFKENNZEICHEN(4054),
    /** Es wurde ein Stichwortendezeichen an Stelle eines Unterfeldkennzeichens
     * gefunden. */
    STICHWORTENDE_STATT_UFKENNZEICHEN(4055),
    /** Es wurde ein Teilfeldtrennzeichen an Stelle eines Unterfeldkennzeichens
     * gefunden. */
    TEILFELDTRENNUNG_STATT_UFKENNZEICHEN(4056),
    /** Es wurde ein Unterfeldbeginnzeichen an Stelle eines
     * Unterfeldkennzeichens gefunden. Es traten also zwei oder mehrere
     * Unterfeldbeginnzeichen in Folge auf. */
    UNTERFELDBEGINN_STATT_UFKENNZEICHEN(4057),
    /** Das Unterfeldkennzeichen ist kein Zeichen aus den Bereichen a-z, A-Z,
     * 0-9 oder Leerzeichen. */
    UFKENNZEICHEN_FORMATFEHLER(4058),
    /** Im Unterfeld wurde ein Nichtsortierendezeichen ausserhalb eines
     * Nichtsortierbereiches gefunden. */
    FEHLENDER_NICHTSORTIERBEGINN_IN_UFELD(4200),
    /** Im Unterfeld wurde ein Nichtsortierendezeichen ausserhalb eines
     * Nichtsortierbereiches gefunden. */
    FEHLENDER_STICHWORTBEGINN_IN_UFELD(4201),
    /** Im Unterfeld wurde ein Nichtsortierbereich nicht abgeschlossen
     * (fehlendes Nichtsortierendezeichen).*/
    FEHLENDES_NICHTSORTIERENDE_IN_UFELD(4202),
    /** Im Unterfeld wurde ein Stichwortendezeichen ausserhalb eines
     * Stichwortbereiches gefunden. */
    FEHLENDES_STICHWORTENDE_IN_UFELD(4203),
    /** Im Unterfeld wurde ein Nichtsortierbeginnzeichen in einem
     * Nichtsortierbereich gefunden. */
    WIEDERHOLTER_NICHTSORTIERBEGINN_IN_UFELD(4204),
    /** Im Unterfeld wurde ein Stichwortbeginnzeichen in einem
     * Stichwortgebinnbereich gefunden. */
    WIEDERHOLTER_STICHWORTBEGINN_IN_UFELD(4205);

    /**
     * Map from error code to ErrorCodes object.
     */
    private static final Map<Integer, ErrorCode> LOOKUP =
        new HashMap<Integer, ErrorCode>();

    /**
     * Base number for range of automatically generated error codes.
     */
    private static final int AUTOMATIC_ERROR_CODE_BASE = 5000;

    /**
     * Generates reversing mapping from error code to ErrorCodes.
     */
    static {
        for (ErrorCode errorCode : EnumSet.allOf(ErrorCode.class)) {
            LOOKUP.put(errorCode.getCode(), errorCode);
        }
    }
    /**
     * Error code of ErrorCodes instance.
     */
    private int errorCode;

    /**
     * Resource bundle with error messages.
     */
    private static Properties messages;

    /**
     * Use dynamically generated error code.
     */
    private ErrorCode() {
        this(-1);
    }

    /**
     * Constructs ErrorCodes object for error code.
     * @param code int
     */
    private ErrorCode(final int code) {
        this.errorCode = code;
    }

    /**
     * Returns the error code for the object.
     * @return int
     */
    public int getCode() {
        if (errorCode == -1) {
            return AUTOMATIC_ERROR_CODE_BASE + this.ordinal();
        }
        else {
            return errorCode;
        }
    }

    /**
     * Returns error message for ErrorCodes instance.
     * @return die entsprechende Fehlernachricht
     */
    public String getMessage() {
        return getMessages().getProperty(this.name());
    }

    /**
     * Returns the ErrorCodes object for specific errorCode.
     * @param errorCode Integer
     * @return ErrorCodes object
     */
    public static ErrorCode get(final int errorCode) {
        return LOOKUP.get(errorCode);
    }

    /**
     * Loads error messages associated with error codes.
     * @return Properties instance containing error messages
     */
    private static Properties getMessages() {
        if (messages == null) {
            messages = new Properties();
            try {
                InputStream in = ErrorCode.class.getClassLoader(
                        ).getResourceAsStream(
                        "de/kobv/mable/error-messages.xml");
                messages.loadFromXML(in);
            }
            catch (IOException ioe) {
                // TODO do something? fatal error, should not happen
            }
        }
        return messages;
    }

}
