/*
 * This file is part of Mable+, a program for checking MAB data for errors.
 *
 * Copyright (C) 2008, 2011-2012 Kooperativer Bibliotheksverbund
 * Berlin-Brandenburg (KOBV) <http://www.kobv.de>,
 * im Konrad-Zuse-Zentrum für Informationstechnik
 * Berlin (ZIB) <http://www.zib.de>, Takustr. 7, D-14195 Berlin-Dahlem
 * Author(s) Jens Schwidder, <schwidder(at)zib.de>,
 *           Pascal-Nicolas Becker, <becker(at)zib.de>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package de.kobv.mable.cli;

import de.ddb.charset.MabCharset;
import de.kobv.mable.common.MableProperties;
import de.kobv.mable.kernels.CheckKernel;
import de.kobv.mable.mab.parser.MabDatasetCounter;
import java.io.*;
import org.apache.commons.io.FilenameUtils;
import org.apache.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * Generates error report for MAB file.
 *
 * @author Jens Schwidder <schwidder(at)zib.de>
 *
 * TODO recognize MABXML and MAB2 files reliably
 * TODO different progress monitoring if no datasetCount (for MABXML)
 * TODO refactor
 * TODO separate code for progress monitoring (move to parent class?)
 * TODO am Ende angelegte Dateien auflisten
 */
public final class CheckCommand extends AbstractMableCommand {

    /**
     * Logger for class.
     */
    private static final Logger LOG = Logger.getLogger(CheckCommand.class);

    /**
     * Option for enabling fast parser.
     */
    private boolean fastParser;

    /**
     * Default name of Spring XML configuration file.
     */
    private static final String DEFAULT_CONFIG_FILE =
            "de/kobv/mable/cli/spring-config-nosql.xml";

    private String configFile = null;

    /**
     * Executes check on MAB file.
     */
    @Override
    public void execute() {
        ApplicationContext context =
            new ClassPathXmlApplicationContext(getConfigFile());

        // Laden des Analyzers.
        Reader source;

        int totalDatasets;

        InputStream in;

        try {
            source = new FileReader(getFilename());
            if (!isQuietMode()) {
                // TODO refactor
                System.out.println(getMessages().getString("mable.cli.prepare"));
            }

            if (!getFilename().endsWith(".xml")) {
                MabDatasetCounter counter = new MabDatasetCounter();
                counter.setSource(source);
                counter.parse();
                totalDatasets = counter.getDatasetCounter();
            }
            else {
                totalDatasets = 0;
            }

            MableProperties.setProperty("dataset.count", totalDatasets);
            MableProperties.setProperty("filename", getFilename());

            in = new FileInputStream(getFilename());

            // TODO refactor? Move into CheckKernel?
            if (isUtf8encoded()) {
                System.out.println("Not using MAB charset.");
                source = new InputStreamReader(in);
            }
            else {
                source = new InputStreamReader(in, new MabCharset());
            }
        }
        catch (FileNotFoundException fnfe) {
            System.err.println(getMessages().getString("error.fileNotFound"));
            return;
        }
        catch (IOException ioe) {
            LOG.error("Error reading file", ioe);
            return;
        }

        CheckKernel kernel = context.getBean(CheckKernel.class);

        kernel.setQuietMode(isQuietMode());

        try {
            kernel.setTotalDatasets(totalDatasets);
            kernel.setFilename(getFilename());
            String extension = FilenameUtils.getExtension(getFilename());
            if (extension.contains("xml")) {
                kernel.setMabxmlInput(true);
                kernel.setStream(in);
            }
            else if (isFastParser()) {
                kernel.setStream(in);
                kernel.setFastMode(true);
            }
            else {
                kernel.setSource(source); // TODO refactor? eliminate?
                kernel.setFastMode(false);
            }
            kernel.start();
        }
        catch (Exception e) {
            LOG.error("Unexpected exception", e);
        }
    }

    /**
     * Returns status of fast parser option.
     * @return true - use fast parser; false - use default parser
     */
    public boolean isFastParser() {
        return fastParser;
    }

    /**
     * Setter for enabling/disabling fast parser.
     * @param enabled true - enable fast parser; false - use default parser
     */
    public void setFastParser(final boolean enabled) {
        this.fastParser = enabled;
    }

    public void setConfigPath(final String path) {
        this.configFile = path;
    }

    private String getConfigFile() {
        if (configFile != null) {
            return configFile;
        }
        else {
            return DEFAULT_CONFIG_FILE;
        }
    }

}
